import itertools
import random
from src.turtlegfx_datagen.prompts.codeedit_specs import *

def sample_spec_combinations(specs_list, n_combine, n_combs_sample):
    """
    Sample combinations of specifications based on the given `specs_list`, `n_combine`, and `n_combs_sample`.

    This function creates combinations of specification classes and then generates all possible
    combinations of specifications from those classes. Finally, it samples `n_combs_sample` combinations.

    Args:
    - specs_list (list): List of specification class names as strings.
    - n_combine (int): Number of specification classes to combine.
    - n_combs_sample (int): Number of combinations to sample.

    Returns:
    - list: A list of lists, where each inner list contains specification values.
    
    Example: sample_spec_combinations(["Add", "Remove", "Scale"], 2, 10)
        [
            [AddSpecs.SPEC_ADD_MOVE, RemoveSpecs.SPEC_REMOVE_INSIDE_LOOP],
            [AddSpecs.SPEC_ADD_MOVE, ScaleSpecs.SPEC_SCALEUP],
            [RemoveSpecs.SPEC_REMOVE_INSIDE_LOOP, ScaleSpecs.SPEC_SCALEUP],
            ...
        ]
    """
    # Generate all combinations of specification classes
    all_class_combs = enum_all_class_combs(specs_list, n_combine)
    
    all_specs_combs = []
    for class_combs in all_class_combs:
        # Convert class names to actual class objects
        spec_classes = [globals()[f"{cls_name}Specs"] for cls_name in class_combs]
        # Generate all combinations of specifications for the current class combination
        all_specs_combs.extend(enum_all_spec_combs(spec_classes))
    
    # Sample n_combs_sample combinations from all_specs_combs
    sampled_specs_combs = random.choices(all_specs_combs, k=n_combs_sample)

    return sampled_specs_combs

def enum_all_class_combs(categories, n_classes_combine):
    """
    Generate all combinations of the given categories (Enum classes).

    Args:
        categories (list): List of category names as strings.
        n_classes_combine (int): Number of categories to combine.

    Returns:
        list: A list of lists, where each inner list contains category names.

    Example:
        Input: categories = ["Add", "Remove", "Scale"], n_classes_combine = 2
        Output: 
        [
            ["Add", "Remove"],
            ["Add", "Scale"],
            ["Remove", "Scale"]
        ]
    """
    return list(itertools.combinations(categories, n_classes_combine))

def enum_all_spec_combs(categories):
    """
    Generate all combinations of specifications from the given categories (Enum classes).

    Args:
        categories (list): List of Enum classes representing specification categories.

    Returns:
        list: A list of tuples, where each tuple contains specification values.

    Example:
        Input: categories = [AddSpecs, RemoveSpecs]
        Output: 
        [
            (AddSpecs.SPEC_ADD_MOVE, RemoveSpecs.SPEC_REMOVE_INSIDE_LOOP),
            (AddSpecs.SPEC_ADD_MOVE, RemoveSpecs.SPEC_REMOVE_FINAL_MOVE),
            (AddSpecs.SPEC_ADD_TURN, RemoveSpecs.SPEC_REMOVE_INSIDE_LOOP),
            (AddSpecs.SPEC_ADD_TURN, RemoveSpecs.SPEC_REMOVE_FINAL_MOVE),
            ...
        ]
    """
    return list(itertools.product(*categories))