#!/bin/bash
#SBATCH --job-name=postprocess
#SBATCH --partition=spyder
#SBATCH -c 64
#SBATCH -t 4-00:00
#SBATCH --mem=100G
#SBATCH --output=codegen_postprocess_%j.out
#SBATCH --error=codegen_postprocess_%j.err

# Function to display usage information
usage() {
    echo "Usage: $0 [-c <experiment_config_file>] [-p <part>]"
    echo "  -c <config>     Experiment-specific config file (optional)"
    echo "  -p <part>       Part number to process"
    exit 1
}

# Parse command line arguments
while getopts ":c:p:" opt; do
    case ${opt} in
        c )
            CONFIG_FILE=$OPTARG
            ;;
        p )
            PART=$OPTARG
            ;;
        \? )
            usage
            ;;
    esac
done

# Check if part number is provided
if [ -z "$PART" ]; then
    echo "Error: Part number (-p) must be specified"
    usage
fi

# Set default config file if not provided
if [ -z "$CONFIG_FILE" ]; then
    CONFIG_FILE="exps/datasetgen/scripts/configs/default_config.sh"
fi

# Check if config file exists
if [ ! -f "$CONFIG_FILE" ]; then
    echo "Error: Experiment config file '$CONFIG_FILE' not found."
    exit 1
fi

# Source config files
source "$CONFIG_FILE"
source "exps/datasetgen/scripts/configs/global_config.sh"

# Override paths with part number
INPUT_PATH="${CODEGEN_RESPONSES_DIR}/${CODEGEN_PROMPT_NAME}__${CODEGEN_MODEL_NAME_SHORT}__${DATE}_part_${PART}.json"
OUTPUT_PATH="${CODEGEN_POSTPROCESS_DIR}/${CODEGEN_PROMPT_NAME}__${CODEGEN_MODEL_NAME_SHORT}__${DATE}_part_${PART}.json"

# Print configuration
echo "Configuration:"
echo "- Config file: $CONFIG_FILE"
echo "- Part number: $PART"
echo "- Input path: ${INPUT_PATH}"
echo "- Output path: ${OUTPUT_PATH}"

# Use the part number as the display number
DISPLAY_NUM=$((1000 + PART))
Xvfb :${DISPLAY_NUM} -screen 0 1600x1200x16 &
XVFB_PID=$!
export DISPLAY=:${DISPLAY_NUM}.0

echo "Starting postprocessing for part ${PART}"

${PYTHON_TURTLE} src/turtlegfx_datagen/codegen/postprocess.py \
    --input_path "${INPUT_PATH}" \
    --output_path "${OUTPUT_PATH}" \
    --output_type success \
    --max_workers 48

if [ $? -ne 0 ]; then
    echo "Error: Python script failed to execute properly."
    kill $XVFB_PID
    exit 1
fi

kill $XVFB_PID

echo "CODEGEN: Postprocessing complete for part ${PART}. Output saved to ${OUTPUT_PATH}"
