#!/bin/bash

############################################################
# Check dependencies
############################################################
if ! command -v yq &> /dev/null; then
    echo "Error: yq is not installed. Please install it first."
    exit 1
fi

############################################################
# Logging
############################################################
log() {
    echo "$(tput setaf 0)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_warning() {
    echo "$(tput setaf 3)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_done() {
    echo "$(tput setaf 2)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

log_error() {
    echo "$(tput setaf 1)$(date '+%Y-%m-%d %H:%M:%S') $@$(tput sgr0)"
}

############################################################
# Load configs
############################################################
CONFIG_DIR="exps/eval_vlms/scripts/configs"

# Using regex to match YAML files
CONFIG_FILES=$(find "$CONFIG_DIR" -regex ".*/.*\.yaml" ! -name "default_config.yaml")

if [ -z "$CONFIG_FILES" ]; then
    log_error "No config files found in $CONFIG_DIR"
    exit 1
fi

############################################################
# Submit jobs
############################################################
submit_job() {
    local model_key=$1
    local config_file=$2
    
    # Get model configuration
    local gpu_type=$(yq -r ".[\"$model_key\"].gpu_type" "$config_file")
    local gpu_count=$(yq -r ".[\"$model_key\"].gpu_count" "$config_file")
    local model_name=$(yq -r ".[\"$model_key\"].model_name" "$config_file")
    local model_family=$(yq -r ".[\"$model_key\"].model_family" "$config_file")
    local prompt_file=$(yq -r ".[\"$model_key\"].prompt_file" "$config_file")
    local response_file=$(yq -r ".[\"$model_key\"].response_file" "$config_file")
    local mem=$((gpu_count * 64))G
    local cpu_count=$((gpu_count * 8))
    
    # Skip if gpu_count is 0 (e.g., for OpenAI models)
    if [ "$gpu_count" -eq 0 ]; then
        log_warning "Skipping $model_key (API-based model)"
        return
    fi
    
    # Check if output file already exists
    if [ -f "$response_file" ]; then
        log_warning "Skipping $model_key (output file exists)"
        return
    fi

    log "Model key: $model_key"
    log "Model name: $model_name"
    log "Model family: $model_family"
    log "Prompt file: $prompt_file"
    log "Response file: $response_file"
    log "Config file: $config_file"
    log "GPU type: $gpu_type"
    log "GPU count: $gpu_count"
    log "Memory: $mem"
    log "CPU count: $cpu_count"
    
    # Submit job
    if ! sbatch -p ${gpu_type} \
        --gres=gpu:${gpu_count} \
        --mem=${mem} \
        -c ${cpu_count} \
        exps/eval_vlms/scripts/build_responses_vlms_yaml.sh \
        -m ${model_key} \
        -c ${config_file}; then
        log_error "Failed to submit job for ${model_key}"
    else
        log_done "Successfully submitted job for ${model_key} with ${gpu_count} GPUs"
    fi
}

# Process each config file
for config_file in $CONFIG_FILES; do
    log "Processing config file: $config_file"
    
    # Get all model keys from the current YAML file
    model_keys=$(yq -r 'keys | .[]' "$config_file")
    
    # Submit jobs for all models in this config
    for model_key in $model_keys; do
        submit_job "$model_key" "$config_file"
    done
done 