import json
import argparse

"""
Usage: 
python src/llmutils/convert_json_into_jsonl.py \
    --input_file <input_file> \
    --output_file <output_file>
"""


def convert_json_to_jsonl(input_file, output_file):
    """
    Converts a JSON file to JSON Lines (JSONL) format and writes to an output file.

    Args:
        input_file (str): Path to the input JSON file.
        output_file (str): Path to the output JSONL file.
    
    Example:
        input: [{"id": 1, "name": "Alice"}, {"id": 2, "name": "Bob"}]
        output: 
            {"id": 1, "name": "Alice"}
            {"id": 2, "name": "Bob"}
    """
    with open(input_file, 'r') as f:
        data = json.load(f)
    
    # assert that output_file is ``.jsonl``
    if not output_file.endswith('.jsonl'):
        raise ValueError(f"Output file must have a .jsonl extension. Got {output_file}")

    with open(output_file, 'w') as f:
        for element in data:
            json.dump(element, f)
            f.write('\n')

def main():
    """
    Main function to parse arguments and call the JSON formatting functions.
    """
    parser = argparse.ArgumentParser(
        description='Format a JSON file by wrapping its contents in square brackets or convert it to JSONL format.'
    )
    parser.add_argument('--input_file', type=str, help='Path to the input JSON file.')
    parser.add_argument('--output_file', type=str, help='Path to the output JSON file.')

    args = parser.parse_args()

    convert_json_to_jsonl(args.input_file, args.output_file)
    print(f"JSONL file has been written to {args.output_file}")

if __name__ == "__main__":
    main()