import unittest
import json
import tempfile
import os
from src.turtlegfx_datagen.utils.data_merge import merge_data

class TestDataMerge(unittest.TestCase):
    def test_merge_data(self):
        data_parts = [
            [{"id": "aa", "name": "adf"}, {"id": "bb", "name": "xyz"}],
            [1, 2, 3],
            ["a", "b", "c"]
        ]
        expected_result = [
            {"id": "aa", "name": "adf"},
            {"id": "bb", "name": "xyz"},
            1, 2, 3,
            "a", "b", "c"
        ]
        self.assertEqual(merge_data(data_parts), expected_result)

    def test_merge_data_single_part(self):
        data_parts = [[1, 2, 3]]
        with self.assertRaises(AssertionError):
            merge_data(data_parts)

    def test_merge_data_empty_parts(self):
        data_parts = [[], []]
        self.assertEqual(merge_data(data_parts), [])

    def test_main_functionality(self):
        # Create temporary input files
        input_data = [
            [{"id": "1", "name": "Alice"}],
            [{"id": "2", "name": "Bob"}],
            [{"id": "3", "name": "Charlie"}]
        ]
        
        with tempfile.TemporaryDirectory() as temp_dir:
            input_files = []
            for i, data in enumerate(input_data):
                file_path = os.path.join(temp_dir, f"input_{i}.json")
                with open(file_path, "w") as f:
                    json.dump(data, f)
                input_files.append(file_path)
            
            # Create output file path
            output_file = os.path.join(temp_dir, "output.json")
            
            # Run the main function
            import sys
            import io
            from src.turtlegfx_datagen.utils.data_merge import main
            
            sys.argv = [
                "data_merge.py",
                "--input_pattern", os.path.join(temp_dir, "input_*.json"),
                "--output_path", output_file
            ]
            
            # Capture stdout to avoid printing during tests
            captured_output = io.StringIO()
            sys.stdout = captured_output
            
            main()
            
            # Reset stdout
            sys.stdout = sys.__stdout__
            
            # Check if the output file exists and contains the correct merged data
            self.assertTrue(os.path.exists(output_file))
            with open(output_file, "r") as f:
                merged_data = json.load(f)
            
            expected_result = [
                {"id": "1", "name": "Alice"},
                {"id": "2", "name": "Bob"},
                {"id": "3", "name": "Charlie"}
            ]
            self.assertEqual(merged_data, expected_result)

if __name__ == "__main__":
    unittest.main()
