import logging
from PIL import Image, ImageGrab

logger = logging.getLogger(__name__)

def screen_to_image(screen):
    img = ImageGrab.grab(bbox=(screen._root.winfo_rootx(),
                               screen._root.winfo_rooty(),
                               screen._root.winfo_rootx() + screen.window_width(),
                               screen._root.winfo_rooty() + screen.window_height()))
    return img


def compare_images(img1, img2, tolerance=1):
    """
    Compare two images with a tolerance for pixel differences.
    Only considers non-white pixels and calculates the percentage of different pixels.
    
    Args:
        img1: First PIL Image
        img2: Second PIL Image
        tolerance: Float 0-100 indicating max percentage of different pixels allowed
    
    Returns:
        bool: True if percentage of different pixels is below tolerance
    """
    if img1.size != img2.size:
        return False
        
    data1 = list(img1.getdata())
    data2 = list(img2.getdata())
    
    # Count non-white pixels in both images
    white = (255, 255, 255)
    non_white_pixels = 0
    different_pixels = 0
    
    for p1, p2 in zip(data1, data2):
        # If either pixel is non-white, include in comparison
        if p1 != white or p2 != white:
            non_white_pixels += 1
            # If pixels are different, count as a difference
            if p1 != p2:
                different_pixels += 1
    
    # If there are no non-white pixels, images are considered empty
    if non_white_pixels == 0:
        return True
    
    # Calculate percentage of different pixels
    difference_percentage = (different_pixels / non_white_pixels) * 100
    
    logger.debug(f"Non-white pixels: {non_white_pixels}")
    logger.debug(f"Different pixels: {different_pixels}")
    logger.debug(f"Difference percentage: {difference_percentage:.2f}%")
    
    return difference_percentage <= tolerance


def images_are_equal_from_path(img1_path, img2_path):
    img1 = Image.open(img1_path)
    img2 = Image.open(img2_path)
    return compare_images(img1, img2)
