import json
import argparse
import glob
import os

def merge_data(data_parts: list) -> list:
    """
    Merge data parts into one list.
    
    Example:
    data_parts = [[{"id": "aa", "name": "adf"}, {"id": "bb", "name": "adf"}], [1, 2, 3]]
    merge_data(data_parts) -> [{"id": "aa", "name": "adf"}, {"id": "bb", "name": "adf"}, 1, 2, 3]
    """
    assert len(data_parts) > 1, "At least two data parts are required"

    merged_data = []
    for data_part in data_parts:
        merged_data.extend(data_part)

    assert len(merged_data) == sum([len(data_part) for data_part in data_parts]), f"Merged data length ({len(merged_data)}) does not match the sum of data parts lengths"
    return merged_data

def main():
    parser = argparse.ArgumentParser(description="Merge JSON data files.")
    parser.add_argument("--input_pattern", required=True, help="Input file pattern (e.g., 'input_*.json')")
    parser.add_argument("--output_path", required=True, help="Output file path")
    args = parser.parse_args()

    input_files = glob.glob(args.input_pattern)
    data_parts = []
    for file_path in input_files:
        with open(file_path, 'r') as f:
            data_parts.append(json.load(f))

    merged_data = merge_data(data_parts)

    with open(args.output_path, 'w') as f:
        json.dump(merged_data, f, indent=2)

    print(f"Merged data saved to {args.output_path}")

if __name__ == "__main__":
    main()
