import torch.nn as nn
import torch.nn.functional as F
from layers import BBB_Linear, BBB_Conv2d
from layers import FlattenLayer,ModuleWrapper
import torch

def conv3x3(in_planes, out_planes, stride=1, dilation=1):
    """3x3 convolution with padding"""
    return BBB_Conv2d(
        in_planes, out_planes, kernel_size=3, stride=stride,
        padding=dilation, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes,stride=1):
    """1x1 convolution"""
    return BBB_Conv2d(
        in_planes, out_planes, kernel_size=1, stride=stride, bias=False )



class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1,
                 downsample=None, upsample=None,
                 dilation=1, norm_layer=nn.BatchNorm2d):
        super(BasicBlock, self).__init__()

        self.conv1 = (
            conv3x3(inplanes, planes, stride)
        )
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.upsample = upsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)
        elif self.upsample is not None:
            identity = self.upsample(x)

        out += identity
        out = self.relu(out)

        return out





class ResNetClassifier(nn.Module):
    block = BasicBlock
    num_blocks = [2, 2, 2, 2]

    def __init__(self,outputs, inputs):
        super(ResNetClassifier, self).__init__()

        num_blocks = self.num_blocks
        self.norm_layer = nn.BatchNorm2d
        self.inputs =inputs
        self.num_classes = outputs
        self.nf = nf = 32

        self.conv1 = BBB_Conv2d(self.inputs,  nf * 1, kernel_size=3, stride=1,bias=False)
        self.bn1 = self.norm_layer(nf * 1)
        self.layer1, self.conv_layer1 = self._make_layer(nf * 1, nf * 1, num_blocks[0], stride=1)
        self.layer2, self.conv_layer2 = self._make_layer(nf * 1, nf * 2, num_blocks[1], stride=2)
        self.layer3, self.conv_layer3 = self._make_layer(nf * 2, nf * 4, num_blocks[2], stride=2)
        self.layer4, self.conv_layer4 = self._make_layer(nf * 4, nf * 8, num_blocks[3], stride=2)
        self.flatten = FlattenLayer(nf * 8)
        self.linear = BBB_Linear(nf * 8, self.num_classes)


    def _make_layer(self, nf_in, nf_out , num_blocks, stride):
        norm_layer = self.norm_layer
        block = self.block
        downsample = None
        self.conv = None
        if stride != 1 or nf_in != nf_out:
            self.conv = conv1x1(nf_in, nf_out, stride)

            downsample = nn.Sequential(
               self.conv,
                norm_layer(nf_out),
            )
        layers = [block(
            nf_in, nf_out, stride, downsample=downsample, norm_layer=norm_layer
        )]
        for _ in range(1, num_blocks):
            layers.append(block(nf_out, nf_out, norm_layer=norm_layer))

        return nn.Sequential(*layers),self.conv

    def forward(self, x):

        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, out.shape[2])
        out = out.view(out.size(0), -1)
        out = self.linear(out)


        kl = 0.0
        for module in self.modules():
            if hasattr(module, 'kl_loss'):
                kl = kl + module.kl_loss()


        return out,kl

    def reset_priors_networks(self):

        self.conv1.reset_prior()
        if self.conv_layer1 is not None:
            self.conv_layer1.reset_prior()
        self.conv_layer2.reset_prior()
        self.conv_layer3.reset_prior()
        self.conv_layer4.reset_prior()
        self.linear.reset_prior()


#net = ResNetClassifier(3,10)


