import os
from openai import OpenAI

class ChatGPTClient:
    def __init__(self, api_key=None, model="gpt-3.5-turbo"):
        self.api_key = api_key or os.getenv("OPENAI_API_KEY")
        if not self.api_key:
            raise ValueError("API key is required. Set it as an environment variable or pass it directly.")
        
        self.model = model
        self.client = OpenAI(api_key=self.api_key)
        self.chat_history = []

    def add_message(self, role, content):
        self.chat_history.append({"role": role, "content": content})

    def ask(
        self,
        user_input: str,
        temperature: float = 0,
        remember_history: bool = False,
        max_tokens: int = None,
        logprobs: int = None,
        top_logprobs: int = None,
        **extra_kwargs
    ):
        """
        Send a chat request, optionally asking for token log-probs.

        Returns:
            The single Choice object from the API, so you can do:
                choice.message.content
                choice.logprobs
        """

        if remember_history:
            self.add_message("user", user_input)
        else:
            self.chat_history = [{"role": "user", "content": user_input}]

        # build argument dict
        api_args = {
            "model": self.model,
            "messages": self.chat_history,
            "temperature": temperature,
        }
        if max_tokens is not None:
            api_args["max_tokens"] = max_tokens
        if logprobs is not None:
            api_args["logprobs"] = logprobs
        if top_logprobs is not None:
            api_args["top_logprobs"] = top_logprobs

        # allow any other kwargs (like stop, etc.)
        api_args.update(extra_kwargs)

        # call the API
        response = self.client.chat.completions.create(**api_args)
        reply = response.choices[0]
        self.add_message("assistant", reply.message.content)
        return reply

    def reset(self):
        self.chat_history = []
