# Copyright (c) 2024, NVIDIA CORPORATION.  All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
import logging
import requests
import torch
from typing import Dict, List, Optional

class HuggingFaceModel:
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, pipeline

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)

        if 'Yarn-Llama' in name_or_path:
            model_kwargs = None
        else:
            model_kwargs = {"attn_implementation": "flash_attention_2"}
        
        try:
            self.pipeline = pipeline(
                "text-generation",
                model=name_or_path,
                tokenizer=self.tokenizer,
                trust_remote_code=True,
                device_map="auto",
                torch_dtype=torch.bfloat16,
                model_kwargs=model_kwargs,
            )
        except:
            self.pipeline = None
            self.model = AutoModelForCausalLM.from_pretrained(name_or_path, trust_remote_code=True, device_map="auto", torch_dtype=torch.bfloat16,)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

    def __call__(self, prompt: str, **kwargs) -> Dict[str, List[str]]:
        if self.pipeline is None:
            inputs = self.tokenizer(prompt, return_tensors="pt").to(self.model.device)
            output = self.model.generate(
                **inputs,
                **self.generation_kwargs
            )
            generated_text = self.tokenizer.decode(output[0][inputs.input_ids.shape[1]:], skip_special_tokens=True)
        else:
            output = self.pipeline(text_inputs=prompt, **self.generation_kwargs,)
            assert len(output) == 1
            generated_text = output[0]["generated_text"]
            
        # remove the input form the generated text
        if generated_text.startswith(prompt):
            generated_text = generated_text[len(prompt) :]
                
        if self.stop is not None:
            for s in self.stop:
                generated_text = generated_text.split(s)[0]
        return {'text': [generated_text]}

class MInferenceModel:
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_best_pattern.json')

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

    def __call__(self, prompt: str, **kwargs) -> Dict[str, List[str]]:
        torch.cuda.empty_cache()
        inputs = self.tokenizer(prompt, return_tensors="pt").to(self.model.device)
        output = self.model.generate(
            **inputs,
            **self.generation_kwargs
        )
        generated_text = self.tokenizer.decode(output[0][inputs.input_ids.shape[1]:], skip_special_tokens=True)
            
        # remove the input form the generated text
        if generated_text.startswith(prompt):
            generated_text = generated_text[len(prompt) :]
                
        if self.stop is not None:
            for s in self.stop:
                generated_text = generated_text.split(s)[0]
        return {'text': [generated_text]}

class Dilated1(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_best_pattern.json')

        config.dilated1 = True

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class Dilated2(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_best_pattern.json')

        config.dilated2 = True

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class YiStatic(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_best_pattern.json')

        config.static_pattern = True

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class LlamaStatic(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Llama_3_8B_Instruct_262k_kv_out_v32_fit_o_best_pattern.json')

        config.static_pattern = True

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')
    
class MInferenceOP(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Llama_3_8B_Instruct_262k_kv_out_v32_fit_o_best_pattern.json')

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class MInferenceOPYi(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_best_pattern.json')

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class OPYiHalfV2(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config = AutoConfig.from_pretrained(name_or_path)
        config._attn_implementation = 'eager'
        config.topk = 1
        config.topk_from_layer = 0
        config.topk_dims_file_path = os.path.join(eval_path, 'config/Yi_9B_200k_kv_out_v32_fit_half3_v2_best_pattern.json')

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = minference_patch(self.model)
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class Streaming(MInferenceModel):
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig
        import sys
        import os
        eval_path = os.path.dirname(os.path.dirname(os.path.dirname(__file__)))
        sys.path.append(eval_path)
        from patch import minference_patch, patch_hf

        self.tokenizer = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
        config = AutoConfig.from_pretrained(name_or_path)

        config._attn_implementation = 'eager'

        self.pipeline = None
        self.model = AutoModelForCausalLM.from_pretrained(
            name_or_path,
            torch_dtype='auto',
            device_map='auto',
            config=config,
        )
        self.model = patch_hf(self.model, attn_type="streaming", attn_kwargs={'n_local': 3968, 'n_init': 128})
            
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')

class MambaModel:
    def __init__(self, name_or_path: str, **generation_kwargs) -> None:
        from transformers import AutoTokenizer
        from mamba_ssm.models.mixer_seq_simple import MambaLMHeadModel

        self.tokenizer = AutoTokenizer.from_pretrained("EleutherAI/gpt-neox-20b")
        self.device = "cuda"
        self.model = MambaLMHeadModel.from_pretrained(name_or_path, device=self.device, dtype=torch.bfloat16)
        self.generation_kwargs = generation_kwargs
        self.stop = self.generation_kwargs.pop('stop')
        self.max_genlen = self.generation_kwargs.pop('max_new_tokens')
        self.minp = 0.0

    def __call__(self, prompt: str, **kwargs) -> Dict[str, List[str]]:
        # tokenize
        tokens = self.tokenizer(prompt, return_tensors="pt")
        input_ids = tokens.input_ids.to(self.device)
        max_length = input_ids.shape[1] + self.max_genlen

        # generate
        out = self.model.generate(
            input_ids=input_ids,
            max_length=max_length,
            cg=True,
            return_dict_in_generate=True,
            output_scores=True,
            enable_timing=False,
            **self.generation_kwargs,
        )
        assert len(out.sequences) == 1
        # detok
        return {'text': [self.tokenizer.decode(out.sequences[0][input_ids.shape[1] :])]}