from PIL import Image
import numpy as np
import torch
import torchvision
import cv2
import math
from shapely.geometry import Point, MultiPoint, Polygon

import json
import pickle
from collections import OrderedDict

from re import X
import numpy as np
import torch
from icecream import ic
from pyquaternion import Quaternion
from opencood.utils.common_utils import check_numpy_to_torch

"""""""""""""""
Camera Utils
"""""""""""""""

def load_camera_data(camera_files, preload=True):
    """
    Args:
        camera_files: list, 
            store camera path
        shape : tuple
            (width, height), resize the image, and overcoming the lazy loading.
    Returns:
        camera_data_list: list,
            list of Image, RGB order
    """
    camera_data_list = []
    for camera_file in camera_files:
        camera_data = Image.open(camera_file)
        if preload:
            camera_data = camera_data.copy()
        camera_data_list.append(camera_data)
    return camera_data_list

def load_intrinsic_DAIR_V2X(int_dict):
    # cam_D : [5, ], what'is this...
    # cam_K : [9, ]
    cam_D = int_dict['cam_D']
    cam_K = int_dict['cam_K']
    return np.array(cam_K).reshape(3,3)


"""""""""""""""
Common Utils
"""""""""""""""

def read_json(file_path):
    with open(file_path, 'r') as f:
        data = json.load(f)

    return data


"""""""""""""""
Transformation Utils
"""""""""""""""

def tfm_to_pose(tfm: np.ndarray):
    """
    turn transformation matrix to [x, y, z, roll, yaw, pitch]
    we use radians format.
    tfm is pose in transformation format, and XYZ order, i.e. roll-pitch-yaw
    """
    # There forumlas are designed from x_to_world, but equal to the one below.
    yaw = np.degrees(np.arctan2(tfm[1,0], tfm[0,0])) # clockwise in carla
    roll = np.degrees(np.arctan2(-tfm[2,1], tfm[2,2])) # but counter-clockwise in carla
    pitch = np.degrees(np.arctan2(tfm[2,0], ((tfm[2,1]**2 + tfm[2,2]**2) ** 0.5)) ) # but counter-clockwise in carla


    # These formulas are designed for consistent axis orientation
    # yaw = np.degrees(np.arctan2(tfm[1,0], tfm[0,0])) # clockwise in carla
    # roll = np.degrees(np.arctan2(tfm[2,1], tfm[2,2])) # but counter-clockwise in carla
    # pitch = np.degrees(np.arctan2(-tfm[2,0], ((tfm[2,1]**2 + tfm[2,2]**2) ** 0.5)) ) # but counter-clockwise in carla

    # roll = - roll
    # pitch = - pitch

    x, y, z = tfm[:3,3]
    return([x, y, z, roll, yaw, pitch])

def tfm_to_xycs_torch(tfm: torch.Tensor):
    """
        similar to tfm_to_pose_torch,
        return x/y/cos(yaw)/sin(yaw)
    """
    x = tfm[:,0,3]
    y = tfm[:,1,3]
    
    cos = tfm[:,0,0]
    sin = tfm[:,1,0]

    pose = torch.stack([x,y,cos,sin]).T # (N, 4)

    return pose

def xycs_to_tfm_torch(xycs: torch.Tensor):
    """
        Args: xycs
            [N, 4]
    """
    N = xycs.shape[0]
    tfm = torch.eye(4, device=xycs.device).view(1,4,4).repeat(N,1,1)

    x, y, cos, sin = xycs[:,0], xycs[:,1], xycs[:,2], xycs[:,3]

    tfm[:,0,0] = cos
    tfm[:,0,1] = - sin
    tfm[:,1,0] = sin
    tfm[:,1,1] = cos
    tfm[:,0,3] = x
    tfm[:,1,3] = y

    return tfm

def tfm_to_pose_torch(tfm: torch.Tensor, dof: int):
    """
    turn transformation matrix to [x, y, z, roll, yaw, pitch]
    we use degree format.
    tfm is pose in transformation format, and XYZ order, i.e. roll-pitch-yaw

    Args:
        tfm: [N, 4, 4]
        dof: 3 or 6
    Returns:
        6dof pose: [N, 6]
    """

    # There forumlas are designed from x_to_world, but equal to the one below.
    yaw = torch.rad2deg(torch.atan2(tfm[:,1,0], tfm[:,0,0])) # clockwise in carla
    roll = torch.rad2deg(torch.atan2(-tfm[:,2,1], tfm[:,2,2])) # but counter-clockwise in carla
    pitch = torch.rad2deg(torch.atan2(tfm[:,2,0], (tfm[:,2,1]**2 + tfm[:,2,2]**2) ** 0.5)) # but counter-clockwise in carla

    # These formulas are designed for consistent axis orientation
    # yaw = torch.rad2deg(torch.atan2(tfm[:,1,0], tfm[:,0,0])) # clockwise in carla
    # roll = torch.rad2deg(torch.atan2(tfm[:,2,1], tfm[:,2,2])) # but counter-clockwise in carla
    # pitch = torch.rad2deg(torch.atan2(-tfm[:,2,0], (tfm[:,2,1]**2 + tfm[:,2,2]**2) ** 0.5)) # but counter-clockwise in carla

    # roll = - roll
    # pitch = - pitch

    x = tfm[:,0,3]
    y = tfm[:,1,3]
    z = tfm[:,2,3]
    
    if dof == 6:
        pose = torch.stack([x,y,z,roll,yaw,pitch]).T # (N, 6)
    elif dof == 3:
        pose = torch.stack([x,y,yaw]).T
    else:
        raise("Only support returning 3dof/6dof pose.")

    return pose

def veh_side_rot_and_trans_to_trasnformation_matrix(lidar_to_novatel_json_file,novatel_to_world_json_file):
    matrix = np.empty([4,4])
    rotationA2B = lidar_to_novatel_json_file["transform"]["rotation"]
    translationA2B = lidar_to_novatel_json_file["transform"]["translation"]
    rotationB2C = novatel_to_world_json_file["rotation"]
    translationB2C = novatel_to_world_json_file["translation"]
    rotation,translation = muilt_coord(rotationA2B, translationA2B, rotationB2C, translationB2C)
    matrix[0:3, 0:3] = rotation
    matrix[:, 3][0:3] = np.array(translation)[:, 0]
    matrix[3, 0:3] = 0
    matrix[3, 3] = 1
    
    return matrix

def inf_side_rot_and_trans_to_trasnformation_matrix(json_file,system_error_offset):
    matrix = np.empty([4,4])
    matrix[0:3, 0:3] = json_file["rotation"]
    translation = np.array(json_file["translation"])
    translation[0][0] = translation[0][0] + system_error_offset["delta_x"]
    translation[1][0] = translation[1][0] + system_error_offset["delta_y"]  #为啥有[1][0]??? --> translation是(3,1)的
    matrix[:, 3][0:3] = translation[:, 0]
    matrix[3, 0:3] = 0
    matrix[3, 3] = 1

    return matrix

def rot_and_trans_to_trasnformation_matrix(json_file):
    matrix = np.empty([4,4])
    matrix[0:3, 0:3] = json_file["rotation"]
    matrix[:, 3][0:3] = np.array(json_file["translation"])[:, 0]
    matrix[3, 0:3] = 0
    matrix[3, 3] = 1

    return matrix
