#### script to fit and calculate outlier scores for pair copulas

############################ help functions

### calculate significance
signif_maker <- function(parmethod,
                         incop,
                         discount,
                         tree){

  if(parmethod == "itau"){
    if(abs(rvinecopulib::par_to_ktau(incop)) < 0.1){
      signif <- 0
    } else {
      signif <- min(abs(rvinecopulib::par_to_ktau(incop))/0.6, 1) * discount^(tree-1)
    }

  } else {
    n1 <- 100
    temp <- seq(0.001, 0.999, length.out = n1)
    eval_points <- cbind(rep(temp, n1-1), rep(temp, each = n1-1))
    grid_dense <- sort(rvinecopulib::dbicop(eval_points, incop))
    n_grid <- length(grid_dense)

    min_dense_area <- mean(grid_dense[1:floor(0.05*n_grid)])
    max_dense_area <- mean(grid_dense[ceiling( (0.95)*n_grid):n_grid])
    dens_diff <- max_dense_area - min_dense_area

    # check if copula is too close to independence
    min_diff <- 1
    max_diff <- 4
    if( dens_diff <= min_diff ) {
      signif <- 0
    } else {
      # calculate copula significance, higher trees get discount factor
      signif <- min((dens_diff-min_diff)/(max_diff-min_diff), 1) * discount^(tree-1)

      print(signif)
      # disregard small significance values for faster computation
      signif <- if(signif < 0.01) 0 else signif
    }

  } # end else(parmethod == "itau")

  return(signif)
}

# generate grid where to evaluate
eval_grid_maker <- function(incop,
                            n){

  temp <- seq(1/(n*10), 1-(1/(n*10)), length.out = n)
  eval_points <- cbind(rep(temp, each = n-1), rep(temp, n-1))
  eval_points <- cbind(rep(temp, each = n-1),
                       rvinecopulib::hbicop(eval_points,
                                            incop,
                                            inverse = T,
                                            cond_var = 1))

  # evaluate density at grid points
  return(sort(rvinecopulib::dbicop(eval_points, incop)))
}

# generate score of sample
scores_maker <- function(grid_dense,
                         indata_dense,
                         signif){
  # cap return value
  n_grid <- length(grid_dense)
  thresh <- (n_grid)^-1
  b <- thresh^-1

  # save ordering to revers sort later
  indata_order <- order(indata_dense)
  indata_dense <- indata_dense[indata_order]
  indata_scores <- rep(0, length(indata_dense))

  #calculate scores
  pos <- 1
  temp <- 1
  for(i in 1:length(indata_dense)){
    temp <- Position(function(x) x > indata_dense[i], grid_dense[pos:length(grid_dense)])
    if(is.na(temp)){
      indata_scores[i:length(indata_scores)] <- 0
      break
    } else {
      temp2 <- (temp+pos-2) / n_grid
      if(temp2 < thresh){
        indata_scores[i] <- signif
      } else {
        indata_scores[i] <- -signif*0.99*log(temp2, base = b)
      }
    }
    pos <- pos + temp - 1
  }

  return(indata_scores[order(indata_order)])
}

################################# end help functions

#### function to generate outlier score function
cop_outl_fit <- function(indata,
                         incop,
                         position,
                         lrarg,
                         cond,
                         n_eval,
                         discount,
                         parmethod,
                         calc_fit){

  if(incop$family != "indep"){
    # determine significance of current copula
    signif <- signif_maker(parmethod = parmethod,
                           incop = incop,
                           discount = discount,
                           tree = position[1])

    # if copula significant, calculate sample scores
    # and outlier function for new points
    if(signif > 0){
      grid_dense <- eval_grid_maker(n = n_eval,
                                    incop = incop)

      if(calc_fit){
      indata_scores <- scores_maker(grid_dense = grid_dense,
                                    indata_dense = rvinecopulib::dbicop(indata, incop),
                                    signif = signif)
      } else {
        indata_scores <- c()
      }

      return(list("cop" = incop,
                  "position" = position,
                  "lrarg" = lrarg,
                  "cond" = cond,
                  "signif" = signif,
                  "grid_dense" = grid_dense,
                  "sample_scores" = indata_scores))
    }
  }

  # if not significant, return cop anyways, perhaps needed to calculate
  # arguments of other cops in higher trees
  return(list("cop" = incop,
              "position" = position,
              "signif" = signif))
}

# input argument fit_info is output from cop_outl_fit
cop_outl <- function(indata,
                     fit_info){

  return(list("sample_scores" = scores_maker(grid_dense = fit_info[["grid_dense"]],
                      indata_dense = rvinecopulib::dbicop(indata, fit_info[["cop"]]),
                      signif = fit_info[["signif"]]),
              "signif" = fit_info[["signif"]],
              "position" = fit_info[["position"]],
              "lrarg" = fit_info[["lrarg"]],
              "cond" = fit_info[["cond"]])
    )
}

