import os
from functools import lru_cache, wraps
from hashlib import sha256

import numpy as np
import pandas as pd
import rpy2.rinterface_lib.callbacks
import rpy2.robjects as ro
from rpy2.robjects import pandas2ri, r
from rpy2.robjects.packages import importr

# robjects.numpy2ri.activate()
pandas2ri.activate()
dir_path = os.path.dirname(os.path.realpath(__file__))
r_dir = os.path.join(dir_path, "R")


# Loop through all files in the directory
for file_name in os.listdir(r_dir):
    # Check if the file ends with .R
    if file_name.endswith(".R"):
        # Construct the full path to the file
        full_path = os.path.join(r_dir, file_name)
        # Source the R file
        r["source"](full_path)
rvcbod_fit = ro.globalenv["vcbod_fit"]
rvcbod = ro.globalenv["vcbod"]


def r_callback_console_print(x):
    print(x, end="")


rpy2.rinterface_lib.callbacks.consolewrite_print = r_callback_console_print


def get_model(
    tree_depth: int,
    dependence: bool,
    discount: float = 0.75,
    desc_thresh: float = 0.1,
    cores: int = 4,
    use_dep_scores_only: bool = False,
):
    return RVC_BOD(
        n_cores=cores,
        tree_depth=tree_depth,
        dependence=dependence,
        discount=discount,
        discr_thresh=desc_thresh,
        use_dep_scores_only=use_dep_scores_only,
    )


class RVC_BOD:
    def __init__(
        self,
        tree_depth=1,
        dependence=True,
        discr_thresh=0.1,
        calc_fit=True,
        discount=0.9,
        n_cores=1,
        verbose=True,
        use_dep_scores_only=True,
    ):
        self.trunclvl = tree_depth
        self.dependence = dependence
        self.discr_thresh = discr_thresh
        self.calc_fit = calc_fit
        self.discount = discount
        self.ncores = n_cores
        self.verbose = verbose
        self.use_dep_scores_only = use_dep_scores_only

    def fit(self, data: np.ndarray):
        p_data = pd.DataFrame(data)
        r_data = r["as.matrix"](pandas2ri.py2rpy(p_data))
        self.fitted_vc = rvcbod_fit(
            indata=r_data,
            dependence=self.dependence,
            discr_thresh=self.discr_thresh,
            calc_fit=False,
            n_eval=100,
            discount=self.discount,
            trunclvl=self.trunclvl,
            ncores=self.ncores,
            verbose=True,
        )
        return self

    def predict_point(self, point: np.ndarray):
        p_data = pd.DataFrame(point)
        r_data = r["as.matrix"](pandas2ri.py2rpy(p_data))
        results = rvcbod(
            r_data,
            self.fitted_vc,
            ncores=self.ncores,
        )
        return results[0]
