'''
Script for collecting interaction experience which can be used for task identification.

Example:
python src/py/min_rl/collect_data.py --config src/py/min_rl/config/rl/imlb+sbho+mlb.yaml --sector 8  --steps 24 --outpath logs/obs_data/sec8_24steps.csv
'''
import sys
from pathlib import Path
LIBPATH = str(Path(__file__).parents[1])
sys.path.append(LIBPATH)
from min_rl import env_utils
import os
import numpy as np
import yaml
import click
import gym
import namegenerator
import copy
import shutil
import csv

HEADER = ['actUE_2300A','actUE_2300B','actUE_1800','actUE_850',
          'ipTput_2300A','ipTput_2300B','ipTput_1800','ipTput_850',
          'cellPrb_2300A','cellPrb_2300B','cellPrb_1800','cellPrb_850']

@click.command()
@click.option('--config', type=str, required=True, help='Path to yaml config file')
@click.option('--sector', type=int, required=True, help='Sector id')
@click.option('--steps', type=int, default=24, required=False, help='Number of steps for data collection. Defaults to 24.')
@click.option('--outpath', type=str, required=True, help='Path of CSV file to output')
def run(config, sector, steps, outpath):
    dir_path = os.path.dirname(os.path.realpath(__file__))

    out_dir = os.path.dirname(os.path.realpath(outpath))
    Path(out_dir).mkdir(parents=True, exist_ok=True)

    params = yaml.safe_load(open(config, 'r'))
    env_kwargs = copy.deepcopy(params['env_kwargs'])
    # Ensure we can collect data across all the days in the scenario
    env_kwargs.update({'day': 'all', 'sector': sector, 'n_steps': steps})
    eval_env = gym.make(params['env_id'], **env_kwargs)

    # The following lines can be used to enable extremely verbose logging
    # log_dir = os.path.join('logs/tmp', namegenerator.gen())
    # Path(log_dir).mkdir(parents=True, exist_ok=True)
    # Enable detailed logging of interactions
    # eval_env.snapshot_config(log_dir)

    eval_env.reset()
    print('Writing CSV file to', outpath)
    with open(outpath, 'w') as csv_file:
        csv_writer = csv.writer(csv_file, delimiter=',', quotechar='"', quoting=csv.QUOTE_MINIMAL)
        csv_writer.writerow(HEADER)

        for i in range(steps):
            # Use default (baseline) actions by specifying None
            s, r, done, sls_obs = eval_env.step(None)
            # Environments are often configured to aggregate observations over
            # the last n timesteps. Since we are only interested in the curent
            # observation, we take the last 12 elements
            csv_writer.writerow(list(s[-12:]))
            if done:
                break

if __name__ == '__main__':
    run()
