'''
 *
 *     ICTP: Irreducible Cartesian Tensor Potentials
 *
 *        File:  neighbors.py
 *
 *     Authors: Deleted for purposes of anonymity 
 *
 *     Proprietor: Deleted for purposes of anonymity --- PROPRIETARY INFORMATION
 * 
 * The software and its source code contain valuable trade secrets and shall be maintained in
 * confidence and treated as confidential information. The software may only be used for 
 * evaluation and/or testing purposes, unless otherwise explicitly stated in the terms of a
 * license agreement or nondisclosure agreement with the proprietor of the software. 
 * Any unauthorized publication, transfer to third parties, or duplication of the object or
 * source code---either totally or in part---is strictly prohibited.
 *
 *     Copyright (c) 2024 Proprietor: Deleted for purposes of anonymity
 *     All Rights Reserved.
 *
 * THE PROPRIETOR DISCLAIMS ALL WARRANTIES, EITHER EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO IMPLIED WARRANTIES OF MERCHANTABILITY 
 * AND FITNESS FOR A PARTICULAR PURPOSE AND THE WARRANTY AGAINST LATENT 
 * DEFECTS, WITH RESPECT TO THE PROGRAM AND ANY ACCOMPANYING DOCUMENTATION. 
 * 
 * NO LIABILITY FOR CONSEQUENTIAL DAMAGES:
 * IN NO EVENT SHALL THE PROPRIETOR OR ANY OF ITS SUBSIDIARIES BE 
 * LIABLE FOR ANY DAMAGES WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES
 * FOR LOSS OF BUSINESS PROFITS, BUSINESS INTERRUPTION, LOSS OF INFORMATION, OR
 * OTHER PECUNIARY LOSS AND INDIRECT, CONSEQUENTIAL, INCIDENTAL,
 * ECONOMIC OR PUNITIVE DAMAGES) ARISING OUT OF THE USE OF OR INABILITY
 * TO USE THIS PROGRAM, EVEN IF the proprietor HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * For purposes of anonymity, the identity of the proprietor is not given herewith. 
 * The identity of the proprietor will be given once the review of the 
 * conference submission is completed. 
 *
 * THIS HEADER MAY NOT BE EXTRACTED OR MODIFIED IN ANY WAY.
 *
'''
import numpy as np
from typing import *

from ase import Atoms

from matscipy.neighbours import neighbour_list


def get_matscipy_neighbors(positions: np.ndarray,
                           cell: np.ndarray,
                           pbc: Union[List[bool], bool],
                           r_cutoff: float,
                           skin: float = 0.0,
                           eps: float = 1e-8,
                           buffer: float = 1.0,
                           **config: Any) -> Tuple[np.ndarray, np.ndarray]:
    """Computes neighbor lists using 'matscipy'.

    Args:
        positions (np.ndarray): Atom positions.
        cell (np.ndarray): Unit cell.
        pbc (Union[List[bool], bool]): Periodic boundaries.
        r_cutoff (float): Cutoff radius.
        skin (float, optional): Skin distance. Defaults to 0.
        eps (float, optional): Small number to check if no cell is provided. Defaults to 1e-8.
        buffer (float, optional): Small buffer for stability if no cell is provided. Defaults to 1.0.

    Returns:
        Tuple[np.ndarray, np.ndarray]: Neighbors (edge indices) and shift vectors (the number of cell boundaries crossed by the bond between atoms).
    """
    atoms = Atoms(positions=positions, cell=cell, pbc=pbc)

    # Add cell if none is present (volume = 0)
    if atoms.cell.volume < eps:
        # max values - min values along xyz augmented by small buffer for stability
        new_cell = np.ptp(atoms.positions, axis=0) + buffer
        # Set cell and center
        atoms.set_cell(new_cell, scale_atoms=False)
        atoms.center()

    # Compute neighborhood
    idx_i, idx_j, S = neighbour_list("ijS", atoms, r_cutoff + skin)
    edge_idx = np.stack([idx_i, idx_j])
    offset = S @ cell

    return edge_idx, offset
