"""
@author: anonymized
@license: anonymized
@contact: anonymized
@software: BNN
@file: resent.py
@time: 2020/10/17 14:22
@desc:Binary Neural Network Optimization
"""

import sys
import math
import random
import torch
import torch.nn as nn
import torch.nn.functional as F
from torchvision.transforms import transforms
from .modules import BConv, MultiBConv, GhostSign, GhostBNSign


__all__ = ['boolnet18', 'boolnet34']


def init_model(model):
    for m in model.modules():

        if isinstance(m, (nn.Conv2d)):
            nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')

            if isinstance(m, BConv):
                m.weight.data.clamp_(-0.99, 0.99)

            if m.bias is not None:
                nn.init.constant_(m.bias, 0)

        elif isinstance(m, (BConv, MultiBConv)):
            nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')

            m.weight.data.mul_(1e-2)
            #m.weight.data.clamp_(-0.5, 0.5)

            if m.temperature is not None:
                nn.init.constant_(m.temperature, 1)

        elif isinstance(m, GhostSign):

            if m.temperature is not None:
                nn.init.constant_(m.temperature, 1)

            #if m.length_1 is not None:
            #   nn.init.constant_(m.length_1, 1)

            #if m.length_2 is not None:
            #   nn.init.constant_(m.length_2, 1)

        elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm, nn.InstanceNorm2d)):
            if m.weight is not None:
                nn.init.constant_(m.weight, 1)

            if m.bias is not None:
                nn.init.constant_(m.bias, 0)

            if m.running_mean is not None:
                nn.init.constant_(m.running_mean, 0)

            if m.running_var is not None:
                nn.init.constant_(m.running_var, 1)



class Conv1x1(nn.Module):
    def __init__(self,
                 in_channels = 3,
                 out_channels = 64,
                 stride = 1,
                 dilation = 1,
                 groups = 1,
                 bias = False):
        super(Conv1x1, self).__init__()

        self.Conv1x1 = nn.Conv2d(in_channels, out_channels, 1, stride = stride, padding = 0, groups = groups, bias = bias)

    def forward(self, x):
        return self.Conv1x1(x)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_channels=16, out_channels=16, stride=1, dilation=1, bias=False,
                 downsample=None, last_block=False, max_slices=4, max_pool=False):
        super(BasicBlock, self).__init__()
        self.max_slices = max_slices

        stochastic = False

        self.conv_slices_1 = 2**(random.randint(0,int(math.log(max_slices)/math.log(2)))) if stochastic else self.max_slices  #specify an int even number to tell the slices to be used(less than max_slices)
        self.conv_slices_2 = 2**(random.randint(0,int(math.log(max_slices)/math.log(2)))) if stochastic else self.max_slices  #specify an int even number to tell the slices to be used(less than max_slices)

        self.stride = stride

        self.in_channels = in_channels

        self.conv1 = MultiBConv(self.in_channels, out_channels, 3, 1 if max_pool else stride, dilation, dilation,
                                groups=self.conv_slices_1, bias=bias, wb=True)
        self.conv2 = MultiBConv(out_channels, out_channels, 3, 1, dilation, dilation,
                                groups=self.conv_slices_2, bias=bias, wb=True)

        self.last_block = last_block

        self.bn1 = nn.BatchNorm2d(out_channels)
        if last_block:
            self.bn2 = nn.BatchNorm2d(out_channels)
            self.bn3 = nn.BatchNorm2d(out_channels)
        else:
            self.bn2 = nn.BatchNorm2d(out_channels)

        self.ghostsign1 = GhostSign(out_channels, slices = self.max_slices)
        self.ghostsign2 = GhostSign(out_channels, slices = self.max_slices)

        self.downsample = downsample

        if stride == 2 and max_pool:
            self.maxpool_if_stride = nn.MaxPool2d(kernel_size=2, stride=2, padding=0)
        else:
            self.maxpool_if_stride = lambda x: x

        self.stride = stride

    def forward(self, x):
        N, S, C, H, W = x.size()
        
        x = self.conv1(x)
        x = self.maxpool_if_stride(x)
        x = self.bn1(x)
        x = self.ghostsign1(x)

        z = self.conv2(x)
        z = self.bn2(z)
        
        if not self.last_block:
            z = self.ghostsign2(z)
        else:
            z = z + self.bn3(x.mean(1))
        
        return z

class BoolNet(nn.Module):
    def __init__(self, block, layers, num_classes=10, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 max_slices = 4, local_adaptive_shifting=False,
                 replace_stride_with_maxpool=False):

        super(BoolNet, self).__init__()
        expansion = 1

        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]

        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))

        print("replace_stride_with_dilation:{}".format(replace_stride_with_dilation))
        scale = 1
        self.groups = groups
        self.base_width = width_per_group
        self.max_slices = max_slices
        self.local_adaptive_shifting = local_adaptive_shifting
        self.replace_stride_with_maxpool = replace_stride_with_maxpool

        self.inplanes = 64*scale
        self.dilation = 1

        self.conv1 = nn.Conv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3, bias = False)

        self.maxpool = nn.Sequential(
            nn.BatchNorm2d(self.inplanes),
            nn.MaxPool2d(kernel_size = 3, stride = 2, padding = 1)
        )

        if self.local_adaptive_shifting:
            self.norm_and_sign = GhostBNSign(self.inplanes, slices=self.max_slices, binary=False)
        else:
            self.norm_and_sign = nn.Sequential(
                nn.BatchNorm2d(self.inplanes),
                GhostSign(self.inplanes, slices=self.max_slices)
            )

        self.layer1 = self._make_layer(block, 64*scale, layers[0], max_slices = self.max_slices)

        self.layer2 = self._make_layer(block, 128*scale, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0], max_slices = self.max_slices)

        self.layer3 = self._make_layer(block, 256*scale, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1], max_slices = self.max_slices)

        self.layer4 = self._make_layer(block, 512*scale, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2], max_slices = self.max_slices,
                                       adapt_last_block=True)

        self.prelu2 = nn.PReLU(512 * scale * block.expansion)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))

        self.fc = nn.Linear(512 * scale * block.expansion, num_classes)

        # Zero-initialize the last BN in each residual branch,

        # so that the residual branch starts with zeros, and each residual block behaves like an identity.

        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677

        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 1e-8)

        init_model(self)

        self.distillation_loss = 0

        self.name = 'boolnet'

    def _make_layer(self, block, planes, blocks, stride=1, dilate=False, max_slices=4, adapt_last_block=False):

        previous_dilation = self.dilation

        if dilate:

            self.dilation *= stride

            stride = 1

        layers = []

        layers.append(block(self.inplanes, planes, stride, previous_dilation, False, None,
                            max_slices=max_slices, max_pool=self.replace_stride_with_maxpool))

        self.inplanes = planes * block.expansion

        for _ in range(1, blocks-1):
            layers.append(block(self.inplanes, planes, 1, self.dilation, False, None,
                                max_slices=max_slices, max_pool=self.replace_stride_with_maxpool))

        # last block in the last stage is slightly different:
        layers.append(block(self.inplanes, planes, 1, self.dilation, False, None,
                            max_slices=max_slices, max_pool=self.replace_stride_with_maxpool,
                            last_block=adapt_last_block))

        return nn.Sequential(*layers)

    def forward(self, x):
        out = []

        x = self.conv1(x)
        x = self.maxpool(x)
        x = self.norm_and_sign(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.prelu2(x)

        out = self.fc(self.avgpool(x).contiguous().view(x.size(0), -1))

        return out

class boolnet18(BoolNet):
    """Constructs a resnet18 model with Binarized weight and activation.  
    
    Args:
      num_classes(int): an int number used to identify the num of classes, default 10
      inflate(int): an int number used to control the width of a singel convolution layer, default 4(4*16)
    """
    def __init__(self, **kwargs):
        super(boolnet18, self).__init__(block=BasicBlock, layers=(2, 2, 2, 2), **kwargs)
        self.name = 'boolnet18'

class boolnet34(BoolNet):
    """Constructs a resnet34 model with Binarized weight and activation.  
    
    Args:
      num_classes(int): an int number used to identify the num of classes, default 10
      inflate(int): an int number used to control the width of a singel convolution layer, default 4(4*16)
    
    """
    def __init__(self, **kwargs):
        super(boolnet34, self).__init__(block=BasicBlock, layers=(3, 4, 6, 3), **kwargs)
        self.name = 'boolnet34'
