#!/bin/bash

# Interactive image comparison Dashboard - Select and compare images from Taylor, HiCache modes in web interface
# Usage: ./compare_images.sh [options]

# Default settings
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
INTERVAL=6
RESULTS_DIR="${REPO_ROOT}/results"
PORT=8080

# Help information
show_help() {
    echo "Usage: $0 [options]"
    echo ""
    echo "Options:"
    echo "  -i, --interval N  Specify interval value [default: 6]"
    echo "  -d, --dir DIR     Specify results directory path [default: ${RESULTS_DIR}]"
    echo "  -p, --port PORT   Specify HTTP server port [default: 8080]"
    echo "  -h, --help        Show help information"
    echo ""
    echo "Description:"
    echo "  Start interactive web dashboard, can select images for comparison in browser"
    echo ""
    echo "Examples:"
    echo "  $0                       # Start dashboard with default settings"
    echo "  $0 -i 4                   # Start with interval_4 directory"
    echo "  $0 --port 9000            # Start with port 9000"
}

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        -i|--interval)
            INTERVAL="$2"
            shift 2
            ;;
        -d|--dir)
            RESULTS_DIR="$2"
            shift 2
            ;;
        -p|--port)
            PORT="$2"
            shift 2
            ;;
        -h|--help)
            show_help
            exit 0
            ;;
        -*)
            echo "Error: Unknown option '$1'"
            show_help
            exit 1
            ;;
        *)
            echo "Error: Unsupported parameter '$1'"
            show_help
            exit 1
            ;;
    esac
done

if command -v realpath >/dev/null 2>&1; then
    RESULTS_DIR="$(realpath "$RESULTS_DIR")"
else
    RESULTS_DIR="$(cd "$RESULTS_DIR" 2>/dev/null && pwd)"
fi

if [[ -z "$RESULTS_DIR" || ! -d "$RESULTS_DIR" ]]; then
    echo "Error: Cannot access results directory"
    exit 1
fi

# Validate port number
if ! [[ "$PORT" =~ ^[0-9]+$ ]] || [[ "$PORT" -lt 1024 ]] || [[ "$PORT" -gt 65535 ]]; then
    echo "Error: Port number must be a number between 1024-65535"
    exit 1
fi

# Show configuration information
echo "================================="
echo "Interactive image comparison Dashboard configuration:"
echo "Interval: $INTERVAL"
echo "Results directory: $RESULTS_DIR"
echo "Port: $PORT"
echo "================================="

# Scan available images
scan_images() {
    local taylor_dir="${RESULTS_DIR}/taylor/interval_${INTERVAL}"
    local hicache_dir="${RESULTS_DIR}/hicache/interval_${INTERVAL}"
    
    declare -A images
    
    # Scan Taylor directory
    if [[ -d "$taylor_dir" ]]; then
        for img in "$taylor_dir"/img_*.jpg; do
            if [[ -f "$img" ]]; then
                local num=$(basename "$img" .jpg | sed 's/img_//')
                images["$num"]="taylor"
            fi
        done
    fi
    
    # Scan HiCache directory
    if [[ -d "$hicache_dir" ]]; then
        for img in "$hicache_dir"/img_*.jpg; do
            if [[ -f "$img" ]]; then
                local num=$(basename "$img" .jpg | sed 's/img_//')
                if [[ -n "${images[$num]}" ]]; then
                    images["$num"]="${images[$num]},hicache"
                else
                    images["$num"]="hicache"
                fi
            fi
        done
    fi
    
    # Output JSON format image information
    echo "{"
    local first=true
    for num in $(printf '%s\n' "${!images[@]}" | sort -n); do
        if [[ "$first" == true ]]; then
            first=false
        else
            echo ","
        fi
        echo -n "    \"$num\": {"
        echo -n "\"modes\": \"${images[$num]}\""
        
        # Check if files exist for each mode
        local taylor_exists="false"
        local hicache_exists="false"
        
        [[ -f "$taylor_dir/img_$num.jpg" ]] && taylor_exists="true"
        [[ -f "$hicache_dir/img_$num.jpg" ]] && hicache_exists="true"
        
        echo -n ", \"taylor\": $taylor_exists"
        echo -n ", \"hicache\": $hicache_exists"
        echo -n "}"
    done
    echo ""
    echo "}"
}

# Create interactive dashboard
create_interactive_dashboard() {
    local html_file="./interactive_dashboard.html"
    local images_json=$(scan_images)
    
    cat > "$html_file" << EOF
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Interactive Image Comparison Dashboard</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }
        
        .container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: linear-gradient(135deg, #2c3e50 0%, #3498db 100%);
            color: white;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            margin-bottom: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        
        .header p {
            font-size: 1.1em;
            opacity: 0.9;
        }
        
        .controls {
            background: white;
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 30px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .controls h2 {
            margin-bottom: 20px;
            color: #2c3e50;
        }
        
        .image-selector {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-bottom: 20px;
        }
        
        .image-option {
            background: #f8f9fa;
            border: 2px solid #dee2e6;
            border-radius: 8px;
            padding: 10px 15px;
            cursor: pointer;
            transition: all 0.3s ease;
            min-width: 80px;
            text-align: center;
        }
        
        .image-option:hover {
            background: #e9ecef;
            border-color: #3498db;
        }
        
        .image-option.selected {
            background: #3498db;
            color: white;
            border-color: #2980b9;
        }
        
        .image-option .modes {
            font-size: 0.8em;
            margin-top: 5px;
            opacity: 0.7;
        }
        
        .comparison-area {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            min-height: 400px;
        }
        
        .comparison-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 30px;
        }
        
        .image-card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            overflow: hidden;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .image-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
        }
        
        .image-header {
            padding: 20px;
            font-size: 1.3em;
            font-weight: bold;
            text-align: center;
            color: white;
        }
        
        .taylor-header { background: linear-gradient(135deg, #ff6b6b, #ff8e8e); }
        .hicache-header { background: linear-gradient(135deg, #4ecdc4, #44a08d); }
        
        .image-container {
            position: relative;
            padding: 20px;
        }
        
        .image-container img {
            width: 100%;
            height: auto;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .image-info {
            padding: 15px 20px;
            background: #f8f9fa;
            font-size: 0.9em;
            color: #6c757d;
        }
        
        .not-found {
            text-align: center;
            padding: 40px;
            color: #e74c3c;
            font-style: italic;
        }
        
        .loading {
            text-align: center;
            padding: 60px;
            color: #6c757d;
        }
        
        .loading::after {
            content: '';
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-left: 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .stats {
            display: flex;
            justify-content: space-around;
            margin: 20px 0;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.5em;
            font-weight: bold;
            color: #3498db;
        }
        
        .refresh-btn {
            background: #3498db;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 1em;
            margin-left: 20px;
            transition: background 0.3s ease;
        }
        
        .refresh-btn:hover {
            background: #2980b9;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🖼️ Interactive Image Comparison Dashboard</h1>
            <p>选择图片编号进行Taylor、HiCache两种模式对比</p>
            <div class="stats">
                <div class="stat-item">
                    <div class="stat-value" id="total-images">-</div>
                    <div>可用图片</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">${INTERVAL}</div>
                    <div>间隔值</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value">2</div>
                    <div>对比模式</div>
                </div>
            </div>
        </div>
        
        <div class="controls">
            <h2>📋 选择图片 <button class="refresh-btn" onclick="refreshImages()">🔄 刷新</button></h2>
            <div class="image-selector" id="image-selector">
                <div class="loading">正在加载图片列表...</div>
            </div>
        </div>
        
        <div class="comparison-area">
            <div id="comparison-content">
                <div class="loading">请选择一张图片开始对比...</div>
            </div>
        </div>
    </div>

    <script>
        // 图片数据
        const imagesData = $images_json;
        let selectedImage = null;
        
        // 初始化页面
        function init() {
            renderImageSelector();
            updateStats();
        }
        
        // 渲染图片选择器
        function renderImageSelector() {
            const selector = document.getElementById('image-selector');
            const imageNumbers = Object.keys(imagesData).sort((a, b) => parseInt(a) - parseInt(b));
            
            if (imageNumbers.length === 0) {
                selector.innerHTML = '<div class="not-found">未找到任何图片文件</div>';
                return;
            }
            
            let html = '';
            imageNumbers.forEach(num => {
                const data = imagesData[num];
                const modes = data.modes.split(',');
                const modesText = modes.map(mode => {
                    const icon = mode === 'taylor' ? '🔥' : '🌊';
                    return icon + mode.charAt(0).toUpperCase();
                }).join(' ');
                
                html += `
                    <div class="image-option" onclick="selectImage('${num}')">
                        <div>img_${num}</div>
                        <div class="modes">${modesText}</div>
                    </div>
                `;
            });
            
            selector.innerHTML = html;
        }
        
        // 选择图片
        function selectImage(num) {
            // 更新选中状态
            document.querySelectorAll('.image-option').forEach(option => {
                option.classList.remove('selected');
            });
            event.target.closest('.image-option').classList.add('selected');
            
            selectedImage = num;
            showComparison(num);
        }
        
        // 显示对比结果
        function showComparison(num) {
            const content = document.getElementById('comparison-content');
            content.innerHTML = '<div class="loading">正在加载图片对比...</div>';
            
            const data = imagesData[num];
            
            setTimeout(() => {
                let html = '<div class="comparison-grid">';
                
                // Taylor模式
                if (data.taylor) {
                    html += `
                        <div class="image-card">
                            <div class="image-header taylor-header">🔥 Taylor 模式</div>
                            <div class="image-container">
                                <img src="${RESULTS_DIR}/taylor/interval_${INTERVAL}/img_${num}.jpg" 
                                     alt="Taylor模式图片" loading="lazy" 
                                     onerror="this.parentElement.innerHTML='<div class=not-found>❌ 图片加载失败</div>'">
                            </div>
                            <div class="image-info">
                                <strong>路径:</strong> ${RESULTS_DIR}/taylor/interval_${INTERVAL}/img_${num}.jpg<br>
                                <strong>模式:</strong> Taylor缓存优化
                            </div>
                        </div>
                    `;
                } else {
                    html += `
                        <div class="image-card">
                            <div class="image-header taylor-header">🔥 Taylor 模式</div>
                            <div class="not-found">❌ 图片文件不存在</div>
                        </div>
                    `;
                }
                
                // HiCache模式
                if (data.hicache) {
                    html += `
                        <div class="image-card">
                            <div class="image-header hicache-header">🌊 HiCache 模式</div>
                            <div class="image-container">
                                <img src="${RESULTS_DIR}/hicache/interval_${INTERVAL}/img_${num}.jpg" 
                                     alt="HiCache模式图片" loading="lazy"
                                     onerror="this.parentElement.innerHTML='<div class=not-found>❌ 图片加载失败</div>'">
                            </div>
                            <div class="image-info">
                                <strong>路径:</strong> ${RESULTS_DIR}/hicache/interval_${INTERVAL}/img_${num}.jpg<br>
                                <strong>模式:</strong> HiCache插值优化
                            </div>
                        </div>
                    `;
                } else {
                    html += `
                        <div class="image-card">
                            <div class="image-header hicache-header">🌊 HiCache 模式</div>
                            <div class="not-found">❌ 图片文件不存在</div>
                        </div>
                    `;
                }
                
                html += '</div>';
                content.innerHTML = html;
            }, 500);
        }
        
        // 刷新图片列表
        function refreshImages() {
            location.reload();
        }
        
        // 更新统计信息
        function updateStats() {
            const totalImages = Object.keys(imagesData).length;
            document.getElementById('total-images').textContent = totalImages;
        }
        
        // 页面加载完成后初始化
        document.addEventListener('DOMContentLoaded', init);
    </script>
</body>
</html>
EOF

    echo "$html_file"
}

# 创建dashboard并启动服务
echo "🌐 创建交互式网页dashboard..."
html_file=$(create_interactive_dashboard)
echo "✅ Dashboard创建完成: $html_file"

# 查找可用端口
while netstat -ln 2>/dev/null | grep -q ":$PORT " || lsof -i :$PORT 2>/dev/null; do
    ((PORT++))
done

echo "📡 启动HTTP服务器，端口: $PORT"

# 启动HTTP服务器
if command -v python3 &> /dev/null; then
    echo "🐍 使用Python3启动服务器..."
    python3 -m http.server $PORT > /dev/null 2>&1 &
    server_pid=$!
elif command -v python &> /dev/null; then
    echo "🐍 使用Python启动服务器..."
    python -m SimpleHTTPServer $PORT > /dev/null 2>&1 &
    server_pid=$!
else
    echo "❌ 未找到Python，无法启动HTTP服务器"
    echo "📂 请手动打开: $html_file"
    exit 1
fi

# 等待服务器启动
sleep 2

echo ""
echo "🎉 交互式Dashboard已启动！"
echo "🌐 请在浏览器中打开: http://localhost:$PORT/interactive_dashboard.html"
echo ""
echo "⚠️  服务器进程ID: $server_pid"
echo "💡 要停止服务器，请运行: kill $server_pid"
echo ""

# 如果是在SSH环境中，提供端口转发建议
if [[ -n "$SSH_CLIENT" || -n "$SSH_TTY" || -n "$SSH_CONNECTION" ]]; then
    echo "🔧 在Cursor SSH中使用："
    echo "   1. 打开端口转发面板"
    echo "   2. 转发本地端口$PORT到远程端口$PORT"
    echo "   3. 然后访问: http://localhost:$PORT/interactive_dashboard.html"
    echo ""
    echo "✨ 功能特点："
    echo "   • 📋 自动扫描所有可用图片"
    echo "   • 🖱️ 点击选择图片进行对比"
    echo "   • 🔄 支持实时刷新"
    echo "   • 📱 响应式设计，支持移动设备"
fi