import numpy as np
import matplotlib.pyplot as plt
import math
import random
from tqdm.notebook import tqdm
# from tqdm import tqdm
from Environment import Environment
from plotsettings import BBOX_INCHES, signature, maximizeWindow, palette, makemarkers, add_percent_formatter, legend, show_and_save, nrows_ncols, violin_or_box_plot, adjust_xticks_subplots, table_to_latex
import os

nb_break_points = 0  #: Default nb of random events

def Bernoulli(p):
    if np.random.rand()<p:
        return 1
    else :
        return 0

class Rho(object):
    def __init__(self, nb_arms, horizon, delta = 0.1):
        self.nb_arms = nb_arms
        self.horizon = horizon
        self.delta = delta

    def __call__(self, t):
        return np.sqrt(self.nb_arms*np.log(self.horizon/self.delta)/t) + self.nb_arms*np.log(self.horizon/self.delta)/t
    
class Rho_hat(object):
    def __init__(self, nb_arms, horizon, delta = 0.1):
        self.nb_arms = nb_arms
        self.horizon = horizon
        self.n_hat = np.log2(horizon)+1
        self.delta = delta

    def __call__(self, t):
        rho = Rho(nb_arms = self.nb_arms, horizon = self.horizon)
        return 6*self.n_hat*np.log(self.horizon/self.delta)*rho(t)

def schedule(n, t, horizon, nb_arms):
    schedule_list = []
    for tau in range(2**n):
        for m in range(n,-1,-1):
            if tau%(2**m) == 0:
                rho = Rho(nb_arms=nb_arms, horizon=horizon)
                # print(rho(t=2**n)/rho(t=2**m))
                if Bernoulli(rho(t=2**n)/rho(t=2**m))==1:
                    algs = tau + t
                    alge = tau+2**m + t - 1
                    schedule_list.append({"alg.s":algs, "alg.e":alge, "order-m":m})
    return schedule_list

class Malg(object):
    def __init__(self, n, t, horizon, nb_arms):
        self.n = n
        self.schedule_list = schedule(n, t, horizon, nb_arms)
    def __call__(self, alg):
        UCB = np.ones(alg.nb_arms, dtype = float)*10000
        for k in range(alg.nb_arms):
            if alg.n[k]>0:
                UCB[k] = np.sum(alg.Z[k][0:alg.n[k]])/alg.n[k]+np.sqrt(2*np.log(2**alg.order_m)/alg.n[k])
        pi = np.argmax(UCB)
        g = np.max(UCB)

        return pi, g

class instance(object):
    def __init__(self, nb_arms , order_m):
        self.order_m = order_m
        self.reward = []
        self.nb_arms = nb_arms
        self.Z = np.zeros((nb_arms, 2**order_m), dtype= int )
        self.n = np.zeros(nb_arms, dtype = int)

  
class Master(object):
    
    def __init__(self, configuration, ENV = Environment, max_nb_random_events = None, C=1, sub_sample=50, sub_sample2=20, path = None):
        self.C = C
        self.sub_sample = sub_sample
        self.sub_sample2 = sub_sample2
        self.path = path
        self.regret_versus_M = None
        self.regret = None
        self.cfg = configuration  #: Configuration dictionnary
        if ENV.experiment != "K":
            self.nb_break_points = self.cfg.get('nb_break_points')  #: How many random events?
            self.horizon = self.cfg['horizon']  #: Horizon (number of time steps)routermode
            print("Time horizon:", self.horizon)
            # list_nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
            self.nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
            # self.__initEnvironments__()
        self.experiment = ENV.experiment
        self.ENV = ENV
        self.repetitions = self.ENV.repetitions  #: Number of repetitions
        print("Number of repetitions:", self.repetitions)
        
        if ENV.experiment != "K":
            if max_nb_random_events is None:
                max_nb_random_events = len(self.cfg['environment']["params"]["changePoints"])
            assert max_nb_random_events > 0, "Error: for GLR_UCB policy the parameter max_nb_random_events should be > 0 but it was given as {}.".format(max_nb_random_events)
            self.nb_segs = max_nb_random_events


        if self.experiment == "M":
            self.experimentForM(path = path)
        elif self.experiment == "T":
            self.experimentForT(path = path)
        elif self.experiment == "K":
            self.experimentForK(path = path)    
        else:
            self.Env = self.ENV.Env 
            self.mu = self.ENV.mu 
            self.mu_max = self.ENV.mu_max
            self.mean_reward, self.reward_std = self.mainAlg(path = path)        
            self.regret_mean = self.getRegret()
            self.regret_std = self.reward_std

        if path is not None:
            self.__saveRegret(path)
            self.__saveFileName(path)

    def __initEnvironments__(self):
        """ Create environments."""
        print(" Create environments ...")
        list_of_means = np.array(self.cfg['environment'][0]["params"]["listOfMeans"])
        change_point = self.cfg['environment'][0]["params"]["changePoints"]
        self.Env = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype=int)
        self.mu = np.zeros((self.nb_arms, self.horizon), dtype=int)
        self.mu_max = np.zeros(self.horizon ,dtype=float)
        segment = 0
        for t in range(self.horizon):
            for k in range(self.nb_arms):
                if segment < len(change_point) - 1:
                    if t == change_point[segment + 1]:
                        segment += 1
                for i in range(self.repetitions):    
                    self.Env[i][k][t] = self.Bernoulli(list_of_means[segment][k])
                self.mu[k][t] = list_of_means[segment][k]
                self.mu_max[t] = np.max(list_of_means[segment,:])
            
                
    
    def mainAlg(self, path = None):
        # table contains a table of rewards for the K arms up to horizon T
        print("Running ... Master")
        A_mem = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype = int)
        X_mem = np.zeros((self.repetitions, self.horizon), dtype = float)
        rho_hat = Rho_hat(nb_arms=self.nb_arms, horizon=self.horizon)

        for r in tqdm(range(self.repetitions)):
            t = 0
            n = 0
            gt = np.zeros(self.horizon, dtype = float)
            PIt = np.zeros(self.horizon, dtype = int)
            Ut = np.zeros(self.horizon, dtype = float)
            Rt = np.zeros(self.horizon, dtype = float)
            while t < self.horizon:
                t_n = t
                MALG = Malg(n=n, t=t, horizon=self.horizon, nb_arms=self.nb_arms)
                backto=[]
                alg = {}
                g_list = []
                pi_list = []
                for i in range(len(MALG.schedule_list)):
                    alg[str(MALG.schedule_list[i])] = instance(self.nb_arms, MALG.schedule_list[i]["order-m"])
                active_instance = MALG.schedule_list[0]
                while t < t_n + 2**n and t < self.horizon:
                    for i in range(len(MALG.schedule_list)):
                        if t == MALG.schedule_list[i]["alg.s"]:
                            if MALG.schedule_list[i]["alg.e"]<=active_instance["alg.e"] and MALG.schedule_list[i]["order-m"]<active_instance["order-m"]:
                                if active_instance not in backto:     
                                    backto.append(active_instance)
                                active_instance = MALG.schedule_list[i]
                            else:
                                active_instance = MALG.schedule_list[i]
                    if t > active_instance["alg.e"]:
                        active_instance = backto[len(backto)-1]
                        backto.pop()
                    I, g = MALG(alg[str(active_instance)])
                    pi_list.append(I)
                    g_list.append(g)
                    gt[t] = g
                    rew = self.Env[r][I][t]
                    Rt[t] = rew 
                    PIt[t] = I
                    A_mem[r][I][t] = 1
                    X_mem[r][t] = rew
                    alg[str(active_instance)].reward.append(rew)
                    n_instance = alg[str(active_instance)].n
                    alg[str(active_instance)].Z[I][n_instance[I]] = rew
                    alg[str(active_instance)].n[I] = alg[str(active_instance)].n[I] + 1
                    Ut[t] = min(g_list)

                    if t == active_instance["alg.e"]:
                        if np.sum(Rt[active_instance["alg.s"]:active_instance["alg.e"]+1])/(2**active_instance["order-m"]) >= Ut[t] + 9*rho_hat(2**active_instance["order-m"]):
                            test1 = False
                        else:
                            test1 = True
                    
                    if (np.sum(gt[t_n:t+1])-np.sum(Rt[t_n:t+1]))/(t-t_n+1) >= 3*rho_hat(t-t_n+1):
                        test2 = False
                    else:
                        test2 = True

                    t = t+1
                    
                    if test1 == False or test2 == False:
                        break
                n = n + 1

        X_mem_cum = np.cumsum(X_mem,axis=1)
        print("Roll data saving ...")
        if path != None:
            filename = path+'/'+'roll_data'+self.__str__()+str(random.randint(100000,999999))+'.csv'
            np.savetxt(filename, X_mem_cum ,delimiter=",")
        X_mean = np.mean(X_mem, axis=0)
        X_std = np.std(X_mem_cum, axis=0)
        A_mem_cum = np.cumsum(A_mem,axis=2)
        A_std = np.std(A_mem_cum, axis=0)
        A_mean = np.mean(A_mem_cum, axis=0)
        self.plot_arm(A_mean, A_std)
        return X_mean, X_std
    

    def getRegret(self):
        return np.cumsum(self.mu_max)-np.cumsum(self.mean_reward)

    def __str__(self):
        return r"Master+UCB1"
    
    def beta(self, nb, delta=None, variant=None):
        if delta is None:
            delta = 1.0 / np.sqrt(self.horizon)
        if variant is not None:
            if variant == 0:
                c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
            elif variant == 1:
                c = -np.log(delta) + np.log(1 + np.log(nb))
        else:
            c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
        if c < 0 or np.isinf(c):
            c = float('+inf')
        return c

    def CD(self,nb, sums):
        s = 0
        while s < nb :  
            if s % self.subSample2 == 1:
                draw1 = s + 1 
                draw2 = (nb - s) + 1
                mu1 = sums[s] / draw1
                mu2 = (sums[nb] - sums[s]) / draw2 
                mu = sums[nb] / nb
                if (draw1 * self.klChange(mu1, mu) + draw2 * self.klChange(mu2, mu)) > self.beta(nb):
                    return True
            s += 1
        return False
    
    def klChange(self, p, q):
        # binary relative entropy
        res = 0
        if p != q:
            if p <= 0:
                p = np.finfo(float).eps
            if p >= 1:
                p = 1 - np.finfo(float).eps
            res = p * np.log(p / q) + (1 - p) * np.log((1 - p) / (1 - q))
        return res
    
    def klUp(self,p,level):
    # KL upper confidence bound:
    # return uM>p such that d(p,uM)=level 
        lM = p 
        uM = np.min([np.min([1, p + np.sqrt(level/2)]),1]) 
        for j in range(16):
            qM = (uM+lM)/2
            if self.klChange(p,qM) > level:
                uM= qM
            else:
                lM=qM
        return uM

    
    def experimentForM(self, path):
        self.regret_versus_M = [] 
        self.std_regret_versus_M = []
        for m in self.ENV.nb_break_points_list:
            plot_dir = path +"/M"+str(m)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.nb_segs = m
            print("Running ...{} for M = {}".format(self.__str__, m))
            self.Env = self.ENV.Env_for_M[m] 
            self.mu = self.ENV.mu_for_M[m] 
            self.mu_max = self.ENV.mu_max_for_M[m]
            self.mean_reward, self.reward_std = self.mainAlg(path = plot_dir)
            regret = self.getRegret()
            regret_std = self.reward_std
            self.regret_versus_M.append(regret[self.horizon-1])
            self.std_regret_versus_M.append(regret_std[self.horizon-1])

    def experimentForT(self, path):
        self.regret_versus_T = [] 
        self.std_regret_versus_T = []
        self.nb_segs = self.cfg.get('nb_break_points')  #: How many random events?

        for T in self.ENV.horizon_list:
            plot_dir = path +"/T"+str(T)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.horizon = T
            print("Running ...{} for T = {}".format(self.__str__, T))

            self.Env = self.ENV.Env_for_T[T] 
            self.mu = self.ENV.mu_for_T[T] 
            self.mu_max = self.ENV.mu_max_for_T[T]
            self.mean_reward, self.reward_std = self.mainAlg(path = plot_dir)
            regret = self.getRegret()
            regret_std = self.reward_std
            self.regret_versus_T.append(regret[self.horizon-1])
            self.std_regret_versus_T.append(regret_std[self.horizon-1])

    def experimentForK(self, path):
        self.regret_versus_K = [] 
        self.std_regret_versus_K = []
        self.nb_break_points = self.cfg[0].get('nb_break_points')  #: How many random events?
        self.horizon = self.cfg[0]['horizon']  #: Horizon (number of time steps)routermode

        regret_for_instance = np.zeros((self.ENV.num_of_instance, self.horizon), dtype=float)
        var_regret_for_instance = np.zeros((self.ENV.num_of_instance, self.horizon), dtype=float)
        for K in self.ENV.nb_arms_list:
            plot_dir = path +"/K"+str(K)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.nb_arms = K
            print("Running ...{} for K = {}".format(self.__str__, K))

            for i in tqdm(range(self.ENV.num_of_instance)):
                self.Env = self.ENV.Env_for_K[i][K-1] 
                self.mu = self.ENV.mu_for_K[i][K-1] 
                self.mu_max = self.ENV.mu_max_for_K[i][K-1] 
                self.mean_reward, self.reward_std = self.mainAlg(path = plot_dir)
                regret_for_instance[i] = self.getRegret()
                var_regret_for_instance[i] = self.reward_std**2
                # regret_std = self.reward_std
            regret = np.mean(regret_for_instance, axis = 0)
            std_regret = np.mean(var_regret_for_instance, axis = 0)/self.repetitions
            self.regret_versus_K.append(regret[self.horizon-1])
            self.std_regret_versus_K.append(std_regret[self.horizon-1])


    def __saveRegret(self, path):
        if self.experiment == "M":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_M_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_M)
            filename = path+'/'+'std_Regrets_versus_M_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_M)
        elif self.experiment == "K":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_K_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_K)
            filename = path+'/'+'std_Regrets_versus_K_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_K)
        elif self.experiment == "T":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_T_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_T)
            filename = path+'/'+'std_Regrets_versus_T_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_T)
        else:
            print("Regret saving ...")
            filename = path+'/'+'Regrets_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_mean)
            filename = path+'/'+'std_Regrets_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_std)

    def __saveFileName(self, path):
        if self.experiment == "M":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_M_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_M_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_M_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_M_'+self.__str__()+'.csv\n')
            f.close()
        elif self.experiment == "K":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_K_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_K_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_K_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_K_'+self.__str__()+'.csv\n')
            f.close()
        elif self.experiment == "T":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_T_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_T_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_T_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_T_'+self.__str__()+'.csv\n')
            f.close()
        else:
            print("FileName saving ...")
            filename = path+'/Regrets_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_'+self.__str__()+'.csv\n')
            f.close()

    def randmax(self, vector, rank=1):
        # returns an argmax uniformly at random among all maximizers
        # (integer, not CartesianIndex)
        sorted_vector = np.sort(vector)[::-1]
        m = sorted_vector[rank - 1]
        indices = np.where(vector == m)[0]
        random_index = indices[np.random.randint(0, len(indices))]
        return random_index

    def Bernoulli(self, p):
        if np.random.rand()<p:
            return 1
        else :
            return 0
        
    def plot_arm(self, mean, std):
        if self.path is not None:
            color = ["#1F77B4", "#FF7F0E", "#2CA02C"]
            formats = ('png', 'pdf', 'eps')
            # plt.rcParams['figure.figsize'] = (12,8)
            # plt.rcParams['figure.dpi'] = 400
            # plt.rcParams['figure.figsize'] = (20,10)
            # plt.rcParams['figure.dpi'] = 200
            plt.rcParams['font.family'] = "sans-serif"
            plt.rcParams['font.sans-serif'] = "DejaVu Sans"
            plt.rcParams['mathtext.fontset'] = "cm"
            plt.rcParams['mathtext.rm'] = "serif"
            fig = plt.figure()
            x = np.linspace(0, self.horizon-1, self.horizon)
            lw = 3
            if self.nb_arms < 4:
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1),color = color[i], lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i],color = color[i], alpha = 0.2)   
            else:        
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1), lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i], alpha = 0.2)  
            plt.grid(True)
            plt.legend(loc = 'upper left')
            plt.xlabel(r"T")
            # plt.xlabel(r"Time steps $t = 1...T$")
            plt.ylabel(r"Number of arm $i$ pulled")
            if self.experiment == "M":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_M="+str(self.nb_segs)
            elif self.experiment == "K":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_K="+str(self.nb_arms)
            elif self.experiment == "T":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_T="+str(self.horizon)
            else: 
                savefig = self.path+'/'+'arm_'+self.__str__()
            show_and_save(False , savefig=savefig, fig=fig, pickleit=None)
        return fig
    
    