import numpy as np
import matplotlib.pyplot as plt
import math
import random
from tqdm import tqdm
from Environment import Environment
from plotsettings import BBOX_INCHES, signature, maximizeWindow, palette, makemarkers, add_percent_formatter, legend, show_and_save, nrows_ncols, violin_or_box_plot, adjust_xticks_subplots, table_to_latex

nb_break_points = 0  #: Default nb of random events
  
class OracleklUCB(object):
    
    def __init__(self, configuration, ENV = Environment, max_nb_random_events = None, path = None):
        self.path = path
        self.regret_versus_M = None
        self.regret = None
        self.cfg = configuration  #: Configuration dictionnary
        self.nb_break_points = self.cfg.get('nb_break_points')  #: How many random events?
        self.horizon = self.cfg['horizon']  #: Horizon (number of time steps)routermode
        print("Time horizon:", self.horizon)
        # list_nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
        self.nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
        # self.__initEnvironments__()
        self.experiment = ENV.experiment
        self.ENV = ENV
        self.repetitions = self.ENV.repetitions  #: Number of repetitions
        print("Number of repetitions:", self.repetitions)
        self.change_point = self.cfg['environment']["params"]["changePoints"]
        if max_nb_random_events is None:
            max_nb_random_events = len(self.cfg['environment']["params"]["changePoints"])
        assert max_nb_random_events > 0, "Error: for Monitored_UCB policy the parameter max_nb_random_events should be > 0 but it was given as {}.".format(max_nb_random_events)
        self.nb_segs = max_nb_random_events


        if self.experiment != "M":
            self.Env = self.ENV.Env 
            self.mu = self.ENV.mu 
            self.mu_max = self.ENV.mu_max
            self.mean_reward, self.reward_std = self.mainAlg()        
            self.regret = self.getRegret()

            self.regret_mean = self.getRegret()
            self.regret_std = self.reward_std

            if path is not None:
                self.__saveRegret(path)
                self.__saveFileName(path)

    def __initEnvironments__(self):
        """ Create environments."""
        print(" Create environments ...")
        list_of_means = np.array(self.cfg['environment'][0]["params"]["listOfMeans"])
        change_point = self.cfg['environment'][0]["params"]["changePoints"]
        self.Env = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype=int)
        self.mu = np.zeros((self.nb_arms, self.horizon), dtype=int)
        self.mu_max = np.zeros(self.horizon ,dtype=float)
        segment = 0
        for t in range(self.horizon):
            for k in range(self.nb_arms):
                if segment < len(change_point) - 1:
                    if t == change_point[segment + 1]:
                        segment += 1
                for i in range(self.repetitions):    
                    self.Env[i][k][t] = self.Bernoulli(list_of_means[segment][k])
                self.mu[k][t] = list_of_means[segment][k]
                self.mu_max[t] = np.max(list_of_means[segment,:])
            
                
    
    def mainAlg(self):
        print("Running ...")
        A_mem = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype = int)
        X_mem = np.zeros((self.repetitions, self.horizon), dtype = float)
        for i in tqdm(range(self.repetitions)):
            # UCB = np.zeros(self.nb_arms,dtype = float)
            # UCB[:] = 10000
            cp_idx = 1
            TotalNumber     = np.zeros(self.nb_arms, dtype = int) # total number of selections 
            TotalSum        = np.zeros( self.nb_arms, dtype = float)
            ChosenArms      = np.zeros(self.horizon, dtype = int)
            ReceivedRewards = np.zeros( self.horizon, dtype = float)
            tloc = 0
            for t in range(self.horizon):
                indices = np.ones(self.nb_arms, dtype = float)
                for k in range(self.nb_arms):
                    if TotalNumber[k]>0:
                        indices[k] = self.klUp(TotalSum[k]/TotalNumber[k], np.log(tloc+1)/TotalNumber[k])
                action = np.argmax(indices)
                rew = self.Env[i][action][t]
                ChosenArms[t]=action 
                ReceivedRewards[t]=rew
                A_mem[i][action][t] = 1
                X_mem[i][t] = rew
                # update everything
                TotalNumber[action]+=1
                TotalSum[action]+=rew
                tloc += 1
                if cp_idx < len(self.change_point):
                    if t == self.change_point[cp_idx]:
                        TotalNumber     = np.zeros(self.nb_arms, dtype = int) # total number of selections 
                        TotalSum        = np.zeros(self.nb_arms, dtype = float)
                        cp_idx += 1
                        tloc = 0
        X_mean = np.mean(X_mem, axis=0)
        X_std = np.std(X_mem, axis=0)
        A_mem_cum = np.cumsum(A_mem,axis=2)
        A_std = np.std(A_mem_cum, axis=0)
        A_mean = np.mean(A_mem_cum, axis=0)
        self.plot_arm(A_mean, A_std)
        return X_mean, X_std

    
    

    def getRegret(self):
        return np.cumsum(self.mu_max)-np.cumsum(self.mean_reward)

    def __str__(self):

        return r"Oracle kl-UCB"
    
    def beta(self, nb, delta=None, variant=None):
        if delta is None:
            delta = 1.0 / np.sqrt(self.horizon)
        if variant is not None:
            if variant == 0:
                c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
            elif variant == 1:
                c = -np.log(delta) + np.log(1 + np.log(nb))
        else:
            c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
        if c < 0 or np.isinf(c):
            c = float('+inf')
        return c

    def CD(self,nb, sums):
        s = 0
        while s < nb :  
            if s % self.subSample2 == 1:
                draw1 = s + 1 
                draw2 = (nb - s) + 1
                mu1 = sums[s] / draw1
                mu2 = (sums[nb] - sums[s]) / draw2 
                mu = sums[nb] / nb
                if (draw1 * self.klChange(mu1, mu) + draw2 * self.klChange(mu2, mu)) > self.beta(nb):
                    return True
            s += 1
        return False
    
    def klChange(self, p, q):
        # binary relative entropy
        res = 0
        if p != q:
            if p <= 0:
                p = np.finfo(float).eps
            if p >= 1:
                p = 1 - np.finfo(float).eps
            res = p * np.log(p / q) + (1 - p) * np.log((1 - p) / (1 - q))
        return res
    
    def klUp(self,p,level):
    # KL upper confidence bound:
    # return uM>p such that d(p,uM)=level 
        lM = p 
        uM = np.min([np.min([1, p + np.sqrt(level/2)]),1]) 
        for j in range(16):
            qM = (uM+lM)/2
            if self.klChange(p,qM) > level:
                uM= qM
            else:
                lM=qM
        return uM

    
    def experimentForM(self, ChangeKnownledge, alpha):
        self.regret_versus_M = [] 
        for m in self.ENV.nb_break_points_list:
            print("Running ...{} for M = {}".format(self.__str__, m))
            self.Env = self.ENV.Env_for_M[m] 
            self.mu = self.ENV.mu_for_M[m] 
            self.mu_max = self.ENV.mu_max_for_M[m]
            if ChangeKnownledge == "known":
                if alpha is not None:
                    self.alpha = alpha
                else:    
                    self.alpha = np.sqrt(m*self.nb_arms*np.log(self.horizon)/self.horizon)
                print("exploration rate = {}".format(self.alpha))
                self.mean_reward = self.mainAlg() 
            else:
                if alpha is not None:
                    self.alpha0 = alpha
                else:
                    self.alpha0 = np.sqrt(np.log(self.horizon)/self.horizon)
                self.mean_reward = self.mainAlgWithIncreasing() 
            regret = self.getRegret()
            self.regret_versus_M.append(regret[self.horizon-1])

    def __saveRegret(self, path):
        print("Regret saving ...")
        filename = path+'/'+'Regrets_'+self.__str__()+'.csv'
        np.savetxt(filename, self.regret_mean)
        filename = path+'/'+'std_Regrets_'+self.__str__()+'.csv'
        np.savetxt(filename, self.regret_std)

    def __saveFileName(self, path):
        print("FileName saving ...")
        filename = path+'/Regrets_file_name.txt'
        f = open(filename,'a')
        f.write('Regrets_'+self.__str__()+'.csv\n')
        f.close()
        filename = path+'/std_Regrets_file_name.txt'
        f = open(filename,'a')
        f.write('std_Regrets_'+self.__str__()+'.csv\n')
        f.close()

    def randmax(self, vector, rank=1):
        # returns an argmax uniformly at random among all maximizers
        # (integer, not CartesianIndex)
        sorted_vector = np.sort(vector)[::-1]
        m = sorted_vector[rank - 1]
        indices = np.where(vector == m)[0]
        random_index = indices[np.random.randint(0, len(indices))]
        return random_index

    def Bernoulli(self, p):
        if np.random.rand()<p:
            return 1
        else :
            return 0
        
    def plot_arm(self, mean, std):
        if self.path is not None:
            color = ["#1F77B4", "#FF7F0E", "#2CA02C"]
            formats = ('png', 'pdf', 'eps')
            # plt.rcParams['figure.figsize'] = (12,8)
            # plt.rcParams['figure.dpi'] = 400
            # plt.rcParams['figure.figsize'] = (20,10)
            # plt.rcParams['figure.dpi'] = 200
            plt.rcParams['font.family'] = "sans-serif"
            plt.rcParams['font.sans-serif'] = "DejaVu Sans"
            plt.rcParams['mathtext.fontset'] = "cm"
            plt.rcParams['mathtext.rm'] = "serif"
            fig = plt.figure()
            x = np.linspace(0, self.horizon-1, self.horizon)
            lw = 3
            if self.nb_arms < 4:
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1),color = color[i], lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i],color = color[i], alpha = 0.2)   
            else:        
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1), lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i], alpha = 0.2)  
            plt.grid(True)
            plt.legend(loc = 'upper left')
            plt.xlabel(r"T")
            # plt.xlabel(r"Time steps $t = 1...T$")
            plt.ylabel(r"Number of arm $i$ pulled")
            savefig = self.path+'/'+'arm_'+self.__str__()
            show_and_save(False , savefig=savefig, fig=fig, pickleit=None)
            plt.close(fig)
        return fig