import numpy as np
import matplotlib.pyplot as plt
import math
import random
from tqdm import tqdm
from Environment import Environment
from plotsettings import BBOX_INCHES, signature, maximizeWindow, palette, makemarkers, add_percent_formatter, legend, show_and_save, nrows_ncols, violin_or_box_plot, adjust_xticks_subplots, table_to_latex
import os

nb_break_points = 0  #: Default nb of random events
  
class DTS(object):
    
    def __init__(self, configuration, ENV = Environment, max_nb_random_events = None, path = None, gamma = None):
        self.path = path
        self.regret_versus_M = None
        self.regret = None
        self.cfg = configuration  #: Configuration dictionnary
        if ENV.experiment != "K":
            self.nb_break_points = self.cfg.get('nb_break_points')  #: How many random events?
            self.horizon = self.cfg['horizon']  #: Horizon (number of time steps)routermode
            print("Time horizon:", self.horizon)
            # list_nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
            self.nb_arms = len(self.cfg['environment']["params"]["listOfMeans"][0])
            # self.__initEnvironments__()
            if gamma == None:
                gamma = 1-np.sqrt(self.nb_break_points/self.horizon)
        self.gamma = gamma

        self.experiment = ENV.experiment
        self.ENV = ENV
        self.repetitions = self.ENV.repetitions  #: Number of repetitions
        print("Number of repetitions:", self.repetitions)
        
        if ENV.experiment != "K":
            if max_nb_random_events is None:
                max_nb_random_events = len(self.cfg['environment']["params"]["changePoints"])
            assert max_nb_random_events > 0, "Error: for GLR_UCB policy the parameter max_nb_random_events should be > 0 but it was given as {}.".format(max_nb_random_events)
            self.nb_segs = max_nb_random_events


        if self.experiment == "M":
            self.experimentForM(gamma, path=path)
        elif self.experiment == "T":
            self.experimentForT(gamma, path=path)
        elif self.experiment == "K":
            self.experimentForK(gamma, path=path)    
        else:
            self.Env = self.ENV.Env 
            self.mu = self.ENV.mu 
            self.mu_max = self.ENV.mu_max
            self.mean_reward, self.reward_std = self.mainAlg(gamma, path=path)        
            self.regret_mean = self.getRegret()
            self.regret_std = self.reward_std

        if path is not None:
            self.__saveRegret(path)
            self.__saveFileName(path)

    def __initEnvironments__(self):
        """ Create environments."""
        print(" Create environments ...")
        list_of_means = np.array(self.cfg['environment'][0]["params"]["listOfMeans"])
        change_point = self.cfg['environment'][0]["params"]["changePoints"]
        self.Env = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype=int)
        self.mu = np.zeros((self.nb_arms, self.horizon), dtype=int)
        self.mu_max = np.zeros(self.horizon ,dtype=float)
        segment = 0
        for t in range(self.horizon):
            for k in range(self.nb_arms):
                if segment < len(change_point) - 1:
                    if t == change_point[segment + 1]:
                        segment += 1
                for i in range(self.repetitions):    
                    self.Env[i][k][t] = self.Bernoulli(list_of_means[segment][k])
                self.mu[k][t] = list_of_means[segment][k]
                self.mu_max[t] = np.max(list_of_means[segment,:])
            
                
    
    def mainAlg(self,gamma=0.75,a=1,b=1,path = None):
        print("Running ... Discounted TS")
        A_mem = np.zeros((self.repetitions, self.nb_arms, self.horizon), dtype = int)
        X_mem = np.zeros((self.repetitions, self.horizon), dtype = float)
        DiscNumber = np.zeros(self.nb_arms, float)
        DiscSum = np.zeros(self.nb_arms, float)
        Time =0
        for i in tqdm(range(self.repetitions)):
            # UCB = np.zeros(self.nb_arms,dtype = float)
            # UCB[:] = 10000
            ChosenArms      = np.zeros(self.horizon, dtype = int)
            ReceivedRewards = np.zeros( self.horizon, dtype = float)
            for t in range(self.horizon):
                indices = np.ones(self.nb_arms, dtype = float)
                for k in range(self.nb_arms):
                    if DiscNumber[k]>0:
                        indices[k] = np.random.beta(a + DiscSum[k], b + DiscNumber[k] - DiscSum[k])
                I = np.argmax(indices)
                # get the reward  
                rew = self.Env[i][I][t]
                ChosenArms[t]=I 
                ReceivedRewards[t]=rew
                # update everything
                DiscNumber = gamma*DiscNumber
                DiscNumber[I]+=1
                DiscSum = gamma*DiscSum
                DiscSum[I]+=rew
                X_mem[i][t] = rew
                A_mem[i][I][t] = 1
                Time = 1+gamma*Time
        X_mem_cum = np.cumsum(X_mem,axis=1)
        print("Roll data saving ...")
        if path != None:
            filename = path+'/'+'roll_data'+self.__str__()+str(random.randint(100000,999999))+'.csv'
            np.savetxt(filename, X_mem_cum ,delimiter=",")
        X_mean = np.mean(X_mem, axis=0)
        X_std = np.std(X_mem_cum, axis=0)
        A_mem_cum = np.cumsum(A_mem,axis=2)
        A_std = np.std(A_mem_cum, axis=0)
        A_mean = np.mean(A_mem_cum, axis=0)
        self.plot_arm(A_mean, A_std)
        return X_mean, X_std
    
    

    def getRegret(self):
        return np.cumsum(self.mu_max)-np.cumsum(self.mean_reward)

    def __str__(self):
        return r"Discounted-TS"
    
    def beta(self, nb, delta=None, variant=None):
        if delta is None:
            delta = 1.0 / np.sqrt(self.horizon)
        if variant is not None:
            if variant == 0:
                c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
            elif variant == 1:
                c = -np.log(delta) + np.log(1 + np.log(nb))
        else:
            c = -np.log(delta) + (3/2) * np.log(nb) + np.log(3)
        if c < 0 or np.isinf(c):
            c = float('+inf')
        return c

    def CD(self,nb, sums):
        s = 0
        while s < nb :  
            if s % self.subSample2 == 1:
                draw1 = s + 1 
                draw2 = (nb - s) + 1
                mu1 = sums[s] / draw1
                mu2 = (sums[nb] - sums[s]) / draw2 
                mu = sums[nb] / nb
                if (draw1 * self.klChange(mu1, mu) + draw2 * self.klChange(mu2, mu)) > self.beta(nb):
                    return True
            s += 1
        return False
    
    def klChange(self, p, q):
        # binary relative entropy
        res = 0
        if p != q:
            if p <= 0:
                p = np.finfo(float).eps
            if p >= 1:
                p = 1 - np.finfo(float).eps
            res = p * np.log(p / q) + (1 - p) * np.log((1 - p) / (1 - q))
        return res
    
    def klUp(self,p,level):
    # KL upper confidence bound:
    # return uM>p such that d(p,uM)=level 
        lM = p 
        uM = np.min([np.min([1, p + np.sqrt(level/2)]),1]) 
        for j in range(16):
            qM = (uM+lM)/2
            if self.klChange(p,qM) > level:
                uM= qM
            else:
                lM=qM
        return uM

    
    def experimentForM(self, gamma, path):
        self.regret_versus_M = [] 
        self.std_regret_versus_M = []
        for m in self.ENV.nb_break_points_list:
            plot_dir = path +"/M"+str(m)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.nb_segs = m
            print("Running ...{} for M = {}".format(self.__str__, m))
            gamma = 1-np.sqrt(self.nb_segs/self.horizon)
            self.Env = self.ENV.Env_for_M[m] 
            self.mu = self.ENV.mu_for_M[m] 
            self.mu_max = self.ENV.mu_max_for_M[m]
            self.mean_reward, self.reward_std = self.mainAlg(gamma, path = plot_dir)
            regret = self.getRegret()
            regret_std = self.reward_std
            self.regret_versus_M.append(regret[self.horizon-1])
            self.std_regret_versus_M.append(regret_std[self.horizon-1])

    def experimentForT(self, gamma, path):
        self.regret_versus_T = [] 
        self.std_regret_versus_T = []
        self.nb_segs = self.cfg.get('nb_break_points')  #: How many random events?

        for T in self.ENV.horizon_list:
            plot_dir = path +"/T"+str(T)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.horizon = T
            print("Running ...{} for T = {}".format(self.__str__, T))
            gamma = 1-np.sqrt(self.nb_segs/self.horizon)

            self.Env = self.ENV.Env_for_T[T] 
            self.mu = self.ENV.mu_for_T[T] 
            self.mu_max = self.ENV.mu_max_for_T[T]
            self.mean_reward, self.reward_std = self.mainAlg(gamma, path = plot_dir)
            regret = self.getRegret()
            regret_std = self.reward_std
            self.regret_versus_T.append(regret[self.horizon-1])
            self.std_regret_versus_T.append(regret_std[self.horizon-1])

    def experimentForK(self, gamma, path):
        self.regret_versus_K = [] 
        self.std_regret_versus_K = []
        self.nb_break_points = self.cfg[0].get('nb_break_points')  #: How many random events?
        self.horizon = self.cfg[0]['horizon']  #: Horizon (number of time steps)routermode

        regret_for_instance = np.zeros((self.ENV.num_of_instance, self.horizon), dtype=float)
        var_regret_for_instance = np.zeros((self.ENV.num_of_instance, self.horizon), dtype=float)
        for K in self.ENV.nb_arms_list:
            plot_dir = path +"/K"+str(K)
            # Create the sub folder
            if os.path.isdir(plot_dir):
                print("{} is already a directory here...".format(plot_dir))
            elif os.path.isfile(plot_dir):
                raise ValueError("[ERROR] {} is a file, cannot use it as a directory !".format(plot_dir))
            else:
                os.mkdir(plot_dir)
            self.nb_arms = K
            print("Running ...{} for K = {}".format(self.__str__, K))
            gamma = 1-np.sqrt(self.nb_break_points/self.horizon)

            for i in tqdm(range(self.ENV.num_of_instance)):
                self.Env = self.ENV.Env_for_K[i][K-1] 
                self.mu = self.ENV.mu_for_K[i][K-1] 
                self.mu_max = self.ENV.mu_max_for_K[i][K-1] 
                self.mean_reward, self.reward_std = self.mainAlg(gamma, path = plot_dir)
 
                regret_for_instance[i] = self.getRegret()
                var_regret_for_instance[i] = self.reward_std**2
                # regret_std = self.reward_std
            regret = np.mean(regret_for_instance, axis = 0)
            std_regret = np.mean(var_regret_for_instance, axis = 0)/self.repetitions
            self.regret_versus_K.append(regret[self.horizon-1])
            self.std_regret_versus_K.append(std_regret[self.horizon-1])


    def __saveRegret(self, path):
        if self.experiment == "M":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_M_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_M)
            filename = path+'/'+'std_Regrets_versus_M_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_M)
        elif self.experiment == "K":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_K_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_K)
            filename = path+'/'+'std_Regrets_versus_K_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_K)
        elif self.experiment == "T":
            print("Regret saving ...")
            filename = path+'/'+'Regrets_versus_T_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_versus_T)
            filename = path+'/'+'std_Regrets_versus_T_'+self.__str__()+'.csv'
            np.savetxt(filename, self.std_regret_versus_T)
        else:
            print("Regret saving ...")
            filename = path+'/'+'Regrets_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_mean)
            filename = path+'/'+'std_Regrets_'+self.__str__()+'.csv'
            np.savetxt(filename, self.regret_std)

    def __saveFileName(self, path):
        if self.experiment == "M":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_M_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_M_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_M_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_M_'+self.__str__()+'.csv\n')
            f.close()
        elif self.experiment == "K":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_K_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_K_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_K_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_K_'+self.__str__()+'.csv\n')
            f.close()
        elif self.experiment == "T":
            print("FileName saving ...")
            filename = path+'/Regrets_versus_T_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_versus_T_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_versus_T_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_versus_T_'+self.__str__()+'.csv\n')
            f.close()
        else:
            print("FileName saving ...")
            filename = path+'/Regrets_file_name.txt'
            f = open(filename,'a')
            f.write('Regrets_'+self.__str__()+'.csv\n')
            f.close()
            filename = path+'/std_Regrets_file_name.txt'
            f = open(filename,'a')
            f.write('std_Regrets_'+self.__str__()+'.csv\n')
            f.close()

    def randmax(self, vector, rank=1):
        # returns an argmax uniformly at random among all maximizers
        # (integer, not CartesianIndex)
        sorted_vector = np.sort(vector)[::-1]
        m = sorted_vector[rank - 1]
        indices = np.where(vector == m)[0]
        random_index = indices[np.random.randint(0, len(indices))]
        return random_index

    def Bernoulli(self, p):
        if np.random.rand()<p:
            return 1
        else :
            return 0
        
    def plot_arm(self, mean, std):
        if self.path is not None:
            color = ["#1F77B4", "#FF7F0E", "#2CA02C"]
            formats = ('png', 'pdf', 'eps')
            # plt.rcParams['figure.figsize'] = (12,8)
            # plt.rcParams['figure.dpi'] = 400
            # plt.rcParams['figure.figsize'] = (20,10)
            # plt.rcParams['figure.dpi'] = 200
            plt.rcParams['font.family'] = "sans-serif"
            plt.rcParams['font.sans-serif'] = "DejaVu Sans"
            plt.rcParams['mathtext.fontset'] = "cm"
            plt.rcParams['mathtext.rm'] = "serif"
            fig = plt.figure()
            x = np.linspace(0, self.horizon-1, self.horizon)
            lw = 3
            if self.nb_arms < 4:
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1),color = color[i], lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i],color = color[i], alpha = 0.2)   
            else:        
                for i in range(self.nb_arms):
                    plt.plot(mean[i], label = "arm {}".format(i+1), lw=lw)
                    plt.fill_between(x, mean[i]-std[i], mean[i]+std[i], alpha = 0.2)  
            plt.grid(True)
            plt.legend(loc = 'upper left')
            plt.xlabel(r"T")
            # plt.xlabel(r"Time steps $t = 1...T$")
            plt.ylabel(r"Number of arm $i$ pulled")
            if self.experiment == "M":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_M="+str(self.nb_segs)
            elif self.experiment == "K":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_K="+str(self.nb_arms)
            elif self.experiment == "T":
                savefig = self.path+'/'+'arm_'+self.__str__()+"_T="+str(self.horizon)
            else: 
                savefig = self.path+'/'+'arm_'+self.__str__()
            show_and_save(False , savefig=savefig, fig=fig, pickleit=None)
        return fig