"""
Full definition of a GPT Language Model, all of it in this single file.
References:
1) the official GPT-2 TensorFlow implementation released by OpenAI:
https://github.com/openai/gpt-2/blob/master/src/model.py
2) huggingface/transformers PyTorch implementation:
https://github.com/huggingface/transformers/blob/main/src/transformers/models/gpt2/modeling_gpt2.py
"""

import math
import inspect
from dataclasses import dataclass

import torch
import torch.nn as nn
from torch.nn import functional as F
from nn.cola_nn import dense_init, factorize
from nn.moe import MoELayer


class LayerNorm(nn.Module):
    """ LayerNorm but with an optional bias. PyTorch doesn't support simply bias=False """
    def __init__(self, ndim, bias):
        super().__init__()
        self.weight = nn.Parameter(torch.ones(ndim))
        self.bias = nn.Parameter(torch.zeros(ndim)) if bias else None

    def forward(self, input):
        return F.layer_norm(input, self.weight.shape, self.weight, self.bias, 1e-5)


class CausalSelfAttention(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.split_qkv = config.split_qkv
        self.axial = config.axial
        self.use_head_btt = config.use_head_btt
        self.head_btt_case_qkv = config.head_btt_case_qkv
        self.head_btt_case_proj = config.head_btt_case_proj
        self.n_head = config.n_head
        self.d_head = config.d_head
        self.d_model = config.d_model
        self.dropout = config.dropout
        self.do_qk_ln = config.do_qk_ln

        d_attn = config.d_head * config.n_head
        if not self.split_qkv:
            # key, query, value projections for all heads, but in a batch
            self.c_attn = nn.Linear(config.d_model, 3 * d_attn, bias=config.bias)
        else:
            self.c_attn_v = nn.Linear(config.d_model, d_attn, bias=config.bias)
            self.c_attn_q = nn.Linear(config.d_model, d_attn, bias=config.bias)
            self.c_attn_k = nn.Linear(config.d_model, d_attn, bias=config.bias)
        # output projection
        self.c_proj = nn.Linear(d_attn, config.d_model, bias=config.bias)
        if self.use_head_btt:
            for la in [self.c_attn_v, self.c_attn_q, self.c_attn_k]:
                la.use_head_btt = self.use_head_btt
                la.n_head, la.d_head = self.n_head, self.d_head
                la.head_btt_case = self.head_btt_case_qkv
            for la in [self.c_proj]:
                la.use_head_btt = self.use_head_btt
                la.n_head, la.d_head = self.n_head, self.d_head
                la.head_btt_case = self.head_btt_case_proj
        dense_init(self.c_proj, zero_init=True)
        # regularization
        self.attn_dropout = nn.Dropout(config.dropout)
        self.resid_dropout = nn.Dropout(config.dropout)

        if self.do_qk_ln:
            self.ln_q = LayerNorm(config.d_head, bias=config.bias)
            self.ln_k = LayerNorm(config.d_head, bias=config.bias)
        # flash attention make GPU go brrrrr but support is only in PyTorch >= 2.0
        self.flash = hasattr(torch.nn.functional, 'scaled_dot_product_attention')
        if not self.flash:
            print("WARNING: using slow attention. Flash Attention requires PyTorch >= 2.0")
            # causal mask to ensure that attention is only applied to the left in the input sequence
            self.register_buffer(
                "bias",
                torch.tril(torch.ones(config.block_size, config.block_size)).view(1, 1, config.block_size, config.block_size))

    def forward(self, x):
        B, T, C = x.size()  # batch size, sequence length, d_model

        # calculate query, key, values for all heads in batch and move head forward to be the batch dim
        if self.split_qkv:
            q = self.c_attn_q(x)
            k = self.c_attn_k(x)
            v = self.c_attn_v(x)
            if self.use_head_btt:
                if self.head_btt_case_qkv in ["(n|d|n|d)", "(d|n|n|d)"]:
                    for out in [q, k, v]:
                        out = out.reshape(B, T, self.d_head, self.n_head).transpose(-1, -2).reshape(B, T, -1)
        else:
            q, k, v = self.c_attn(x).split(self.d_model, dim=2)
        k = k.view(B, T, self.n_head, self.d_head).transpose(1, 2)  # (B, nh, T, hs)
        q = q.view(B, T, self.n_head, self.d_head).transpose(1, 2)  # (B, nh, T, hs)
        v = v.view(B, T, self.n_head, self.d_head).transpose(1, 2)  # (B, nh, T, hs)

        # QK-layer normalization (per-head)
        if self.do_qk_ln:
            q = self.ln_q(q)
            k = self.ln_k(k)

        k = k * (8 / math.sqrt(k.size(-1)))  # muP attention

        if self.axial:
            # reshape (B, nh, T, hs) -> (B, nh, H, W, hs) where H * W = T
            (H, W) = factorize(T, 2)
            q = q.view(B, self.n_head, H, W, self.d_head)
            k = k.view(B, self.n_head, H, W, self.d_head)
            v = v.view(B, self.n_head, H, W, self.d_head)
            y = torch.nn.functional.scaled_dot_product_attention(q, k, v, attn_mask=None,
                                                                 dropout_p=self.dropout if self.training else 0,
                                                                 is_causal=True)  # (B, nh, H, W, hs)
            # swap H and W to perform attention along the other axis
            q = q.transpose(2, 3).contiguous()  # (B, nh, W, H, hs)
            k = k.transpose(2, 3).contiguous()  # (B, nh, W, H, hs)
            v = y.transpose(2, 3).contiguous()  # (B, nh, W, H, hs)
            y = torch.nn.functional.scaled_dot_product_attention(q, k, v, attn_mask=None,
                                                                 dropout_p=self.dropout if self.training else 0,
                                                                 is_causal=True)  # (B, nh, W, H, hs)
            y = y.transpose(2, 3).contiguous()  # (B, nh, H, W, hs)
            y = y.view(B, self.n_head, T, self.d_head)  # (B, nh, T, hs)
            # y = y.transpose(1, 2).contiguous().view(B, T, -1)  # (B, T, nh * hs) re-assemble heads
        else:
            # causal self-attention; Self-attend: (B, nh, T, hs) x (B, nh, hs, T) -> (B, nh, T, T)
            if self.flash:
                # efficient attention using Flash Attention CUDA kernels
                y = torch.nn.functional.scaled_dot_product_attention(q, k, v, attn_mask=None,
                                                                     dropout_p=self.dropout if self.training else 0,
                                                                     is_causal=True)
            else:
                # manual implementation of attention
                att = (q @ k.transpose(-2, -1)) * (1.0 / math.sqrt(k.size(-1)))
                att = att.masked_fill(self.bias[:, :, :T, :T] == 0, float('-inf'))
                att = F.softmax(att, dim=-1)
                att = self.attn_dropout(att)
                y = att @ v  # (B, nh, T, T) x (B, nh, T, hs) -> (B, nh, T, hs)
            # y = y.transpose(1, 2).contiguous().view(B, T, -1)  # re-assemble all head outputs side by side

        y = y.transpose(1, 2).contiguous()  # (B, T, nh, hs)
        if self.use_head_btt:
            if self.head_btt_case_proj in ["(n|d|n|d)", "(n|d|d|n)"]:
                y = y.transpose(-1, -2)
        y = self.c_proj(y.reshape(B * T, -1)).view(B, T, -1)
        y = self.resid_dropout(y)
        return y


class MLP(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.c_fc = nn.Linear(config.d_model, config.ffn_expansion * config.d_model, bias=config.bias)
        self.gelu = nn.GELU()
        self.c_proj = nn.Linear(config.ffn_expansion * config.d_model, config.d_model, bias=config.bias)
        dense_init(self.c_proj, zero_init=True)
        self.dropout = nn.Dropout(config.dropout)

    def forward(self, x):
        x = self.c_fc(x)
        x = self.gelu(x)
        x = self.c_proj(x)
        x = self.dropout(x)
        return x


class Block(nn.Module):
    def __init__(self, config):
        super().__init__()
        self.ln_1 = LayerNorm(config.d_model, bias=config.bias)
        self.attn = CausalSelfAttention(config)
        self.ln_2 = LayerNorm(config.d_model, bias=config.bias)
        if config.num_ffn_experts > 1:
            self.mlp = MoELayer([MLP(config) for _ in range(config.num_ffn_experts)], in_features=config.d_model,
                                out_features=config.d_model, k=config.num_active_ffn_experts)
        else:
            self.mlp = MLP(config)
        # Depth µP precribes 1/√L branch multiplier and lr multiplier
        self.branch_multiplier = (3 / config.n_layer) ** 0.5 # reduces to SP for 3 layer models
        print(f"Depth-µP branch mult: {self.branch_multiplier}")
        for p in self.parameters():
            p.depth_lr_mult = self.branch_multiplier
                

    def forward(self, x):
        x = x + self.attn(self.ln_1(x)) * self.branch_multiplier
        x = x + self.mlp(self.ln_2(x)) * self.branch_multiplier
        return x


@dataclass
class StructGPTConfig:
    block_size: int = 1024
    vocab_size: int = 50304  # GPT-2 vocab_size of 50257, padded up to nearest multiple of 64 for efficiency
    n_layer: int = 12
    n_head: int = 12
    d_head: int = -1
    d_model: int = 768
    d_embd: int = -1
    ffn_expansion: int = 4
    num_ffn_experts: int = 1
    num_active_ffn_experts: int = 2
    dropout: float = 0.0
    bias: bool = True  # True: bias in Linears and LayerNorms, like GPT-2. False: a bit better and faster
    do_qk_ln: bool = False
    split_qkv: bool = False
    use_head_btt: bool = True
    head_btt_case_qkv: str = "(n|d|n|d)"
    head_btt_case_proj: str = "(n|d|n|d)"
    axial: bool = False  # use axial / BTT attention


class StructGPT(nn.Module):
    def __init__(self, **config_args):
        config = StructGPTConfig(**config_args)
        super().__init__()
        assert config.vocab_size is not None
        assert config.block_size is not None
        if config.d_embd == -1:
            config.d_embd = config.d_model
        assert config.d_embd <= config.d_model, 'Makes no sense to have d_embd > d_model'
        if config.d_head == -1:
            assert config.d_model % config.n_head == 0, "d_model must be divisible by n_head"
            config.d_head = config.d_model // config.n_head
        elif config.n_head == -1:
            assert config.d_model % config.d_head == 0, "d_model must be divisible by d_head"
            config.n_head = config.d_model // config.d_head
        self.config = config
        wte = nn.Embedding(config.vocab_size, config.d_embd)  # (vocab_size, d_embd)
        wte.weight.fan_in_dims = (0, )
        wte.weight.fan_out_dims = (1, )
        if config.d_embd != config.d_model:
            wte_upsample = nn.Linear(config.d_embd, config.d_model, bias=False)
            # wte_downsample = nn.Linear(config.d_model, config.d_embd, bias=False)
        else:
            wte_upsample = nn.Identity()
            # wte_downsample = nn.Identity()
        wpe = nn.Embedding(config.block_size, config.d_model)  # (block_size, d_model)
        wpe.weight.fan_in_dims = (0, )
        wpe.weight.fan_out_dims = (1, )
        self.transformer = nn.ModuleDict(
            dict(
                wte=wte,
                wte_upsample=wte_upsample,
                wpe=wpe,
                drop=nn.Dropout(config.dropout),
                h=nn.ModuleList([Block(config) for _ in range(config.n_layer)]),
                ln_f=LayerNorm(config.d_model, bias=config.bias),
            ))
        self.lm_head = nn.Linear(config.d_model, config.vocab_size, bias=False)
        dense_init(self.lm_head, zero_init=True)
        self.apply(self._init_weights)
        self.fix_init()
        self.steps = 0
        self.hs = [0 for _ in range(config.n_layer + 2)]
        print(f"number of parameters: {self.get_num_params() / 1e6:.2f}M")

    def get_num_params(self, non_embedding=True):
        """
        Return the number of parameters in the model.
        For non-embedding count (default), the position embeddings get subtracted.
        The token embeddings would too, except due to the parameter sharing these
        params are actually used as weights in the final layer, so we include them.
        """
        n_params = sum(p.numel() for p in self.parameters())
        if non_embedding:
            n_params -= self.transformer.wpe.weight.numel()
        return n_params

    def _init_weights(self, module):
        if isinstance(module, nn.Linear):
            # torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)
            if module.bias is not None:
                torch.nn.init.zeros_(module.bias)
        elif isinstance(module, nn.Embedding):
            torch.nn.init.normal_(module.weight, mean=0.0, std=0.02)

    def fix_init(self):
        for m in self.modules():
            if isinstance(m, nn.Linear):
                if hasattr(m.weight, 'zero_init') and m.weight.zero_init:
                    print(f"Skipping zero init: {m}")
                    continue
                else:
                    print(f"Fixing init: {m}")
                    dense_init(m)

    def clear_features(self):
        self.steps = 0
        self.hs = [0 for _ in range(self.config.n_layer + 2)]

    def get_features(self):
        return [(h / self.steps)**0.5 for h in self.hs]

    def set_num_active_experts(self, k):
        for name, module in self.named_modules():
            if hasattr(module, 'num_active_experts'):
                module.prev_num_active_experts = module.num_active_experts
                module.num_active_experts = k
                print(f"Setting num_active_experts={k} for {name}")

    def reset_num_active_experts(self):
        for name, module in self.named_modules():
            if hasattr(module, 'prev_num_active_experts'):
                module.num_active_experts = module.prev_num_active_experts
                print(f"Resetting num_active_experts={module.prev_num_active_experts} for {name}")
    
    def forward(self, idx, targets=None, inference_active_experts=None):
        if inference_active_experts is not None and not self.training:
            self.set_num_active_experts(inference_active_experts)
        device = idx.device
        b, t = idx.size()
        assert t <= self.config.block_size, f"Cannot forward sequence of length {t}, block size is only {self.config.block_size}"
        pos = torch.arange(0, t, dtype=torch.long, device=device)  # shape (t)

        # forward the GPT model itself
        tok_emb = self.transformer.wte(idx)  # token embeddings of shape (b, t, d_embd)
        tok_emb = self.transformer.wte_upsample(tok_emb)  # upsample embeddings
        pos_emb = self.transformer.wpe(pos)  # position embeddings of shape (t, d_model)

        x = self.transformer.drop(tok_emb + pos_emb)
        if not self.training:
            self.hs[0] += torch.mean(x**2).item()
            self.steps += 1
        for i, block in enumerate(self.transformer.h):
            x = block(x)
            if not self.training:
                self.hs[i + 1] += torch.mean(x**2).item()
        # x = self.transformer.wte_downsample(x)  # downsample embeddings
        x = self.transformer.ln_f(x)

        if targets is not None:
            # if we are given some desired targets also calculate the loss
            logits = self.lm_head(x)
            if not self.training:
                self.hs[-1] += torch.mean(logits**2).item()
            loss = F.cross_entropy(logits.view(-1, logits.size(-1)), targets.view(-1), ignore_index=-1)
            aux_losses = []  # 0.01 * feat_mean_sq
            spec_penalties = []
            # load balancing loss
            for name, module in self.named_modules():
                if 'moe_gate' in name:
                    aux_losses.append(module.load_balancing_loss)
                if hasattr(module, 'natural_norm'):
                    spec_penalties.append(module.natural_norm**2)
            dummy_loss = sum((p**2).sum() for p in self.parameters()) * 0
            aux_loss = sum(aux_losses) / len(aux_losses) if aux_losses else torch.tensor(0.0, device=device)
            aux_loss = aux_loss + dummy_loss
            spec_penalty = sum(spec_penalties) / len(spec_penalties) if spec_penalties else torch.tensor(0.0, device=device)
        else:
            # inference-time mini-optimization: only forward the lm_head on the very last position
            logits = self.lm_head(x[:, [-1], :])  # note: using list [-1] to preserve the time dim
            loss = None
            aux_loss = None
            spec_penalty = None
        if inference_active_experts is not None and not self.training:
            self.reset_num_active_experts()
        return logits, loss, aux_loss, spec_penalty

    def crop_block_size(self, block_size):
        # model surgery to decrease the block size if necessary
        # e.g. we may load the GPT2 pretrained model checkpoint (block size 1024)
        # but want to use a smaller block size for some smaller, simpler model
        assert block_size <= self.config.block_size
        self.config.block_size = block_size
        self.transformer.wpe.weight = nn.Parameter(self.transformer.wpe.weight[:block_size])
        for block in self.transformer.h:
            if hasattr(block.attn, 'bias'):
                block.attn.bias = block.attn.bias[:, :, :block_size, :block_size]

    @classmethod
    def from_pretrained(cls, model_type, override_args=None):
        assert model_type in {'gpt2', 'gpt2-medium', 'gpt2-large', 'gpt2-xl'}
        override_args = override_args or {}  # default to empty dict
        # only dropout can be overridden see more notes below
        assert all(k == 'dropout' for k in override_args)
        from transformers import GPT2LMHeadModel
        print("loading weights from pretrained gpt: %s" % model_type)

        # n_layer, n_head and d_model are determined from model_type
        config_args = {
            'gpt2': dict(n_layer=12, n_head=12, d_model=768),  # 124M params
            'gpt2-medium': dict(n_layer=24, n_head=16, d_model=1024),  # 350M params
            'gpt2-large': dict(n_layer=36, n_head=20, d_model=1280),  # 774M params
            'gpt2-xl': dict(n_layer=48, n_head=25, d_model=1600),  # 1558M params
        }[model_type]
        print("forcing vocab_size=50257, block_size=1024, bias=True")
        config_args['vocab_size'] = 50257  # always 50257 for GPT model checkpoints
        config_args['block_size'] = 1024  # always 1024 for GPT model checkpoints
        config_args['bias'] = True  # always True for GPT model checkpoints
        # we can override the dropout rate, if desired
        if 'dropout' in override_args:
            print(f"overriding dropout rate to {override_args['dropout']}")
            config_args['dropout'] = override_args['dropout']
        # create a from-scratch initialized minGPT model
        config = StructGPTConfig(**config_args)
        model = StructGPT(config)
        sd = model.state_dict()
        sd_keys = sd.keys()
        sd_keys = [k for k in sd_keys if not k.endswith('.attn.bias')]  # discard this mask / buffer, not a param

        # init a huggingface/transformers model
        model_hf = GPT2LMHeadModel.from_pretrained(model_type)
        sd_hf = model_hf.state_dict()

        # copy while ensuring all of the parameters are aligned and match in names and shapes
        sd_keys_hf = sd_hf.keys()
        sd_keys_hf = [k for k in sd_keys_hf if not k.endswith('.attn.masked_bias')]  # ignore these, just a buffer
        sd_keys_hf = [k for k in sd_keys_hf if not k.endswith('.attn.bias')]  # same, just the mask (buffer)
        transposed = ['attn.c_attn.weight', 'attn.c_proj.weight', 'mlp.c_fc.weight', 'mlp.c_proj.weight']
        # basically the openai checkpoints use a "Conv1D" module, but we only want to use a vanilla Linear
        # this means that we have to transpose these weights when we import them
        assert len(sd_keys_hf) == len(sd_keys), f"mismatched keys: {len(sd_keys_hf)} != {len(sd_keys)}"
        for k in sd_keys_hf:
            if any(k.endswith(w) for w in transposed):
                # special treatment for the Conv1D weights we need to transpose
                assert sd_hf[k].shape[::-1] == sd[k].shape
                with torch.no_grad():
                    sd[k].copy_(sd_hf[k].t())
            else:
                # vanilla copy over the other parameters
                assert sd_hf[k].shape == sd[k].shape
                with torch.no_grad():
                    sd[k].copy_(sd_hf[k])

        return model

    def configure_optimizers(self, weight_decay, learning_rate, betas, device_type):
        # start with all of the candidate parameters
        param_dict = {pn: p for pn, p in self.named_parameters()}
        # filter out those that do not require grad
        param_dict = {pn: p for pn, p in param_dict.items() if p.requires_grad}
        # create optim groups. Any parameters that is 2D will be weight decayed, otherwise no.
        # i.e. all weight tensors in matmuls + embeddings decay, all biases and layernorms don't.
        decay_params = [p for n, p in param_dict.items() if p.dim() >= 2]
        nodecay_params = [p for n, p in param_dict.items() if p.dim() < 2]
        optim_groups = [{'params': decay_params, 'weight_decay': weight_decay}, {'params': nodecay_params, 'weight_decay': 0.0}]
        num_decay_params = sum(p.numel() for p in decay_params)
        num_nodecay_params = sum(p.numel() for p in nodecay_params)
        print(f"num decayed parameter tensors: {len(decay_params)}, with {num_decay_params:,} parameters")
        print(f"num non-decayed parameter tensors: {len(nodecay_params)}, with {num_nodecay_params:,} parameters")
        # Create AdamW optimizer and use the fused version if it is available
        fused_available = 'fused' in inspect.signature(torch.optim.AdamW).parameters
        use_fused = fused_available and device_type == 'cuda'
        extra_args = dict(fused=True) if use_fused else dict()
        optimizer = torch.optim.AdamW(optim_groups, lr=learning_rate, betas=betas, **extra_args)
        print(f"using fused AdamW: {use_fused}")

        return optimizer

    def estimate_mfu(self, fwdbwd_per_iter, dt):
        """ estimate model flops utilization (MFU) in units of A100 bfloat16 peak FLOPS """
        # first estimate the number of flops we do per iteration.
        # see PaLM paper Appendix B as ref: https://arxiv.org/abs/2204.02311
        N = self.get_num_params()
        cfg = self.config
        L, H, Q, T = cfg.n_layer, cfg.n_head, cfg.d_model // cfg.n_head, cfg.block_size
        flops_per_token = 6 * N + 12 * L * H * Q * T
        flops_per_fwdbwd = flops_per_token * T
        flops_per_iter = flops_per_fwdbwd * fwdbwd_per_iter
        # express our flops throughput as ratio of A100 bfloat16 peak flops
        flops_achieved = flops_per_iter * (1.0 / dt)  # per second
        flops_promised = 312e12  # A100 GPU bfloat16 peak flops is 312 TFLOPS
        mfu = flops_achieved / flops_promised
        return mfu

    @torch.no_grad()
    def generate(self, idx, max_new_tokens, temperature=1.0, top_k=None):
        """
        Take a conditioning sequence of indices idx (LongTensor of shape (b,t)) and complete
        the sequence max_new_tokens times, feeding the predictions back into the model each time.
        Most likely you'll want to make sure to be in model.eval() mode of operation for this.
        """
        for _ in range(max_new_tokens):
            # if the sequence context is growing too long we must crop it at block_size
            idx_cond = idx if idx.size(1) <= self.config.block_size else idx[:, -self.config.block_size:]
            # forward the model to get the logits for the index in the sequence
            logits, _ = self(idx_cond)
            # pluck the logits at the final step and scale by desired temperature
            logits = logits[:, -1, :] / temperature
            # optionally crop the logits to only the top k options
            if top_k is not None:
                v, _ = torch.topk(logits, min(top_k, logits.size(-1)))
                logits[logits < v[:, [-1]]] = -float('Inf')
            # apply softmax to convert logits to (normalized) probabilities
            probs = F.softmax(logits, dim=-1)
            # sample from the distribution
            idx_next = torch.multinomial(probs, num_samples=1)
            # append sampled index to the running sequence and continue
            idx = torch.cat((idx, idx_next), dim=1)

        return idx
