import argparse
import os
import random
import time
import warnings
from enum import Enum
from math import prod

import torch
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.multiprocessing as mp
import torch.nn.parallel
import torch.optim
import torch.utils.data
import torch.utils.data.distributed
import torchvision.datasets as datasets
import torchvision.transforms as transforms
from torch.optim.lr_scheduler import CosineAnnealingLR
from torch.utils.data import Subset

import nn
from nn.cola_nn import cola_parameterize
import wandb
from tqdm import tqdm

parser = argparse.ArgumentParser(description='PyTorch ImageNet Training')
parser.add_argument("--wandb", default=True, action=argparse.BooleanOptionalAction)
parser.add_argument("--wandb_project", default="imagenet_test", type=str, help="Wandb project name")
parser.add_argument('--data', metavar='DIR', nargs='?', default='imagenet', help='path to dataset (default: imagenet)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N', help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=90, type=int, metavar='N', help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N', help='manual epoch number (useful on restarts)')
parser.add_argument(
    '-b', '--batch-size', default=128, type=int, metavar='N', help='mini-batch size (default: 128), this is the total '
    'batch size of all GPUs on the current node when '
    'using Data Parallel or Distributed Data Parallel')

parser.add_argument('--lr', '--learning-rate', default=0.001, type=float, metavar='LR', help='initial learning rate', dest='lr')
parser.add_argument('--wd', '--weight-decay', default=0.01, type=float, metavar='W', help='weight decay (default: 1e-2)',
                    dest='weight_decay')
parser.add_argument('--scheduler', default='cosine', type=str, help='scheduler type')
parser.add_argument('-p', '--log-freq', default=100, type=int, metavar='N', help='print frequency (default: 10)')
parser.add_argument('--resume', default='', type=str, metavar='PATH', help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true', help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true', help='use pre-trained model')
parser.add_argument('--world-size', default=-1, type=int, help='number of nodes for distributed training')
parser.add_argument('--rank', default=-1, type=int, help='node rank for distributed training')
parser.add_argument('--dist-url', default='tcp://224.66.41.62:23456', type=str, help='url used to set up distributed training')
parser.add_argument('--dist-backend', default='nccl', type=str, help='distributed backend')
parser.add_argument('--seed', default=None, type=int, help='seed for initializing training. ')
parser.add_argument('--gpu', default=None, type=int, help='GPU id to use.')
parser.add_argument(
    '--multiprocessing-distributed', action='store_true', help='Use multi-processing distributed training to launch '
    'N processes per node, which has N GPUs. This is the '
    'fastest way to use PyTorch for either single node or '
    'multi node data parallel training')
parser.add_argument('--dummy', action='store_true', help="use fake data to benchmark")

parser.add_argument('--resolution', default=224, type=int, help='inputs will be resolution x resolution')
parser.add_argument('--debug', action='store_true', help="kill run before training")
parser.add_argument('--save_dir', default='imagenet_checkpoints/', type=str, help='directory to save checkpoints')

# Model
parser.add_argument("--model", default="ViT", type=str, help="Type of model")
parser.add_argument("--fixup", action="store_true", help="Use fixup initialization", default=False)
parser.add_argument("--base_width", type=int, default=256)
parser.add_argument("--base_depth", type=int, default=6)
parser.add_argument("--base_heads", type=int, default=4)
parser.add_argument("--base_dim_head", type=int, default=64)

parser.add_argument("--width", type=int, default=None)
parser.add_argument("--depth", type=int, default=None)
parser.add_argument("--heads", type=int, default=None)
parser.add_argument("--dim_head", type=int, default=None)

parser.add_argument("--ffn_expansion", type=int, default=4)
parser.add_argument("--patch_size", type=int, default=16)
parser.add_argument("--in_channels", type=int, default=3)
parser.add_argument("--input_lr_mult", type=float, default=1)
# CoLA
parser.add_argument("--struct", type=str, default="none")

parser.add_argument(
    "--layers",
    type=str,
    default="all_but_last",  # choose from ['ffn', 'attn', 'all_but_last']
)

parser.add_argument("--tt_rank", type=int, default=1)
parser.add_argument("--tt_dim", type=int, default=2)

best_val_acc = 0


def main():
    args = parser.parse_args()
    print(args)

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        cudnn.benchmark = False
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')

    if args.gpu is not None:
        warnings.warn('You have chosen a specific GPU. This will completely '
                      'disable data parallelism.')

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    if torch.cuda.is_available():
        ngpus_per_node = torch.cuda.device_count()
    else:
        ngpus_per_node = 1
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_val_acc
    args.gpu = gpu

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(backend=args.dist_backend, init_method=args.dist_url, world_size=args.world_size, rank=args.rank)
    # create model
    if args.pretrained:
        raise Exception("pretraining not implemented for local models")

    input_shape = (1, 3, args.resolution, args.resolution)
    model_builder = getattr(nn, args.model)
    base_config = dict(dim_in=prod(input_shape), dim_out=1000, depth=args.base_depth, width=args.base_width,
                       heads=args.base_heads, dim_head=args.base_dim_head, ffn_expansion=4, patch_size=args.patch_size,
                       in_channels=args.in_channels, image_size=args.resolution, fixup=args.fixup)

    args.depth = args.base_depth if args.depth is None else args.depth
    args.width = args.base_width if args.width is None else args.width
    if args.heads is None and args.dim_head is None:
        args.heads = args.base_heads if args.heads is None else args.heads
        args.dim_head = args.base_dim_head if args.dim_head is None else args.dim_head
    elif args.heads is not None:
        args.dim_head = args.width // args.heads
    elif args.dim_head is not None:
        args.heads = args.width // args.dim_head
    target_config = dict(dim_in=prod(input_shape), dim_out=1000, depth=args.depth, width=args.width, heads=args.heads,
                         dim_head=args.dim_head, ffn_expansion=4, patch_size=args.patch_size, image_size=args.resolution,
                         in_channels=args.in_channels, fixup=args.fixup)

    def extra_lr_mult_fn(param_name):
        if 'to_patch_embedding' in param_name:
            return args.input_lr_mult
        else:
            return 1

    def zero_init_fn(weight, name):
        return hasattr(weight, 'zero_init') and weight.zero_init

    cola_kwargs = dict(tt_dim=args.tt_dim, tt_rank=args.tt_rank)
    optim_kwargs = dict(weight_decay=args.weight_decay, betas=(0.9, 0.999))
    model, optimizer, cola_info = cola_parameterize(model_builder, base_config=base_config, lr=args.lr, input_shape=input_shape,
                                                    target_config=target_config, struct=args.struct, layer_select_fn=args.layers,
                                                    zero_init_fn=zero_init_fn, extra_lr_mult_fn=extra_lr_mult_fn, device='cuda',
                                                    cola_kwargs=cola_kwargs, optim_kwargs=optim_kwargs)
    if not torch.cuda.is_available() and not torch.backends.mps.is_available():
        print('using CPU, this will be slow')
    elif args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if torch.cuda.is_available():
            if args.gpu is not None:
                torch.cuda.set_device(args.gpu)
                model.cuda(args.gpu)
                # When using a single GPU per process and per
                # DistributedDataParallel, we need to divide the batch size
                # ourselves based on the total number of GPUs of the current node.
                args.batch_size = int(args.batch_size / ngpus_per_node)
                args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
                model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.gpu])
            else:
                model.cuda()
                # DistributedDataParallel will divide and allocate batch_size to all
                # available GPUs if device_ids are not set
                model = torch.nn.parallel.DistributedDataParallel(model)
            print('Running in DDP')
    elif args.gpu is not None and torch.cuda.is_available():
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    elif torch.backends.mps.is_available():
        device = torch.device("mps")
        model = model.to(device)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.model.startswith('alexnet') or args.model.startswith('vgg'):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()
            if hasattr(model.module, 'get_features'):
                model.get_features = model.module.get_features
                model.clear_features = model.module.clear_features
        print('Running in Data Parallel mode')

    if torch.cuda.is_available():
        if args.gpu:
            device = torch.device('cuda:{}'.format(args.gpu))
        else:
            device = torch.device("cuda")
    elif torch.backends.mps.is_available():
        device = torch.device("mps")
    else:
        device = torch.device("cpu")
    # define loss function (criterion), optimizer, and learning rate scheduler
    criterion = torch.nn.CrossEntropyLoss().to(device)

    if args.debug:
        raise Exception("killing. run was in debug mode.")

    # Data loading code
    if args.dummy:
        print("=> Dummy data is used!")
        train_dataset = datasets.FakeData(1281167, (3, args.resolution, args.resolution), 1000, transforms.ToTensor())
        val_dataset = datasets.FakeData(50000, (3, args.resolution, args.resolution), 1000, transforms.ToTensor())
    else:
        traindir = os.path.join(args.data, 'train')
        valdir = os.path.join(args.data, 'val')
        normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225])

        train_dataset = datasets.ImageFolder(
            traindir,
            transforms.Compose([
                transforms.RandomResizedCrop(args.resolution),
                transforms.RandomHorizontalFlip(),
                transforms.ToTensor(),
                normalize,
            ]))

        val_dataset = datasets.ImageFolder(
            valdir,
            transforms.Compose([
                transforms.Resize(256),
                transforms.CenterCrop(args.resolution),
                transforms.ToTensor(),
                normalize,
            ]))

    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
        val_sampler = torch.utils.data.distributed.DistributedSampler(val_dataset, shuffle=False, drop_last=True)
    else:
        train_sampler = None
        val_sampler = None

    train_loader = torch.utils.data.DataLoader(train_dataset, batch_size=args.batch_size, shuffle=(train_sampler is None),
                                               num_workers=args.workers, pin_memory=True, sampler=train_sampler)

    val_loader = torch.utils.data.DataLoader(val_dataset, batch_size=args.batch_size, shuffle=False, num_workers=args.workers,
                                             pin_memory=True, sampler=val_sampler)

    val_batch = next(iter(val_loader))
    """Sets the learning rate to the initial LR decayed by 10 every 30 epochs"""
    # scheduler = StepLR(optimizer, step_size=30, gamma=0.1)
    if args.scheduler == 'none':
        scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=args.epochs, gamma=1.0)
    elif args.scheduler == 'cosine':
        scheduler = CosineAnnealingLR(optimizer=optimizer, T_max=args.epochs * len(train_dataset),
                                      last_epoch=args.start_epoch - 1)
    else:
        raise ValueError(f"Scheduler {args.scheduler} not supported.")

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            elif torch.cuda.is_available():
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint['epoch']
            best_val_acc = checkpoint['best_val_acc']
            if args.gpu is not None:
                # best_val_acc may be from a checkpoint from a different GPU
                best_val_acc = best_val_acc.to(args.gpu)
            model.load_state_dict(checkpoint['state_dict'])
            optimizer.load_state_dict(checkpoint['optimizer'])
            scheduler.load_state_dict(checkpoint['scheduler'])
            print("=> loaded checkpoint '{}' (epoch {})".format(args.resume, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    if args.evaluate:
        validate(val_loader, model, criterion, args)
        return
    if args.wandb and args.rank <= 0:
        config_ = args.__dict__
        config_.update(cola_info)
        wandb.init(
            project=args.wandb_project,
            name=runname(args),
            config=config_,
        )
    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)

        # train for one epoch
        train_acc, train_loss = train(train_loader, model, criterion, optimizer, epoch, device, args, val_batch)

        # evaluate on validation set
        val_acc, val_loss = validate(val_loader, model, criterion, args)

        scheduler.step()

        # remember best acc@1 and save checkpoint
        # is_best = acc1 > best_val_acc
        best_val_acc = max(val_acc, best_val_acc)

        if args.wandb and args.rank <= 0:
            wandb.log({
                "epoch": epoch,
                "train_acc": train_acc,
                "train_loss": train_loss,
                "val_acc": val_acc,
                "val_loss": val_loss,
                "best_acc": best_val_acc,
            })
        if not args.multiprocessing_distributed or (args.multiprocessing_distributed and args.rank % ngpus_per_node == 0):
            save_checkpoint(
                {
                    'epoch': epoch + 1,
                    'arch': args.model,
                    'state_dict': model.state_dict(),
                    'best_val_acc': best_val_acc,
                    'optimizer': optimizer.state_dict(),
                    'scheduler': scheduler.state_dict(),
                    'args': args
                }, save_dir=args.save_dir, filename=get_filename(args))


def rms(x):
    return torch.sqrt(torch.mean(x**2))


def train(train_loader, model, criterion, optimizer, epoch, device, args, val_batch):
    val_imgs = val_batch[0].to(device, non_blocking=True)  # for computing features
    batch_time = AverageMeter('Time', ':6.3f')
    data_time = AverageMeter('Data', ':6.3f')
    losses = AverageMeter('Loss', ':.4e')
    top1 = AverageMeter('Acc@1', ':6.2f')
    top5 = AverageMeter('Acc@5', ':6.2f')
    # progress = ProgressMeter(len(train_loader), [batch_time, data_time, losses, top1, top5], prefix="Epoch: [{}]".format(epoch))

    # switch to train mode
    model.train()

    end = time.time()
    prev_hs = None
    for i, (images, target) in (pbar := tqdm(enumerate(train_loader), total=len(train_loader))):
        # measure data loading time
        data_time.update(time.time() - end)

        # move data to the same device as model
        images = images.to(device, non_blocking=True)
        target = target.to(device, non_blocking=True)

        # compute output
        output = model(images)
        loss = criterion(output, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.log_freq == 0:
            # progress.display(i + 1)
            pbar.set_description(f'Epoch: {epoch} | L: {losses.val:.2f} | Acc: {top1.val:.1f}')
            if args.wandb and args.rank <= 0:
                logs = {"train_acc": top1.val, "train_loss": losses.val}
                if hasattr(model, 'get_features'):
                    model.clear_features()
                    model.eval()  # features only saved in eval mode
                    model(val_imgs)
                    model.train()
                    hs = model.get_features()
                    hs = [torch.cat(h.buffer, dim=0) for h in hs]  # list of tensors
                    if prev_hs is None:
                        prev_hs = hs
                    dhs = [hs[i] - prev_hs[i] for i in range(len(hs))]
                    h_norm = [rms(h) for h in hs]  # should be O(1)
                    h_std = [torch.std(h) for h in hs]
                    dh_norm = [rms(dh) for dh in dhs]  # should be O(1)
                    prev_hs = hs
                    for i in range(len(h_norm)):
                        logs[f'h_{i}'] = h_norm[i].item()
                        logs[f'std_{i}'] = h_std[i].item()
                        logs[f'dh_{i}'] = dh_norm[i].item()
                wandb.log(logs)

    if args.distributed:
        top1.all_reduce()
        losses.all_reduce()
    return top1.avg, losses.avg


def validate(val_loader, model, criterion, args):
    def run_validate(loader, base_progress=0):
        with torch.no_grad():
            end = time.time()
            for i, (images, target) in enumerate(loader):
                i = base_progress + i
                if args.gpu is not None and torch.cuda.is_available():
                    images = images.cuda(args.gpu, non_blocking=True)
                if torch.backends.mps.is_available():
                    images = images.to('mps')
                    target = target.to('mps')
                if torch.cuda.is_available():
                    target = target.cuda(args.gpu, non_blocking=True)

                # compute output
                output = model(images)
                loss = criterion(output, target)

                # measure accuracy and record loss
                acc1, acc5 = accuracy(output, target, topk=(1, 5))
                losses.update(loss.item(), images.size(0))
                top1.update(acc1[0], images.size(0))
                top5.update(acc5[0], images.size(0))

                # measure elapsed time
                batch_time.update(time.time() - end)
                end = time.time()

    batch_time = AverageMeter('Time', ':6.3f', Summary.NONE)
    losses = AverageMeter('Loss', ':.4e', Summary.NONE)
    top1 = AverageMeter('Acc@1', ':6.2f', Summary.AVERAGE)
    top5 = AverageMeter('Acc@5', ':6.2f', Summary.AVERAGE)
    progress = ProgressMeter(
        len(val_loader) + (args.distributed and (len(val_loader.sampler) * args.world_size < len(val_loader.dataset))),
        [batch_time, losses, top1, top5], prefix='Val: ')

    # switch to evaluate mode
    model.eval()

    run_validate(val_loader)
    if args.distributed:
        top1.all_reduce()
        top5.all_reduce()

    if args.distributed and (len(val_loader.sampler) * args.world_size < len(val_loader.dataset)):
        aux_val_dataset = Subset(val_loader.dataset, range(len(val_loader.sampler) * args.world_size, len(val_loader.dataset)))
        aux_val_loader = torch.utils.data.DataLoader(aux_val_dataset, batch_size=args.batch_size, shuffle=False,
                                                     num_workers=args.workers, pin_memory=True)
        run_validate(aux_val_loader, len(val_loader))

    progress.display_summary()

    return top1.avg, losses.avg


def runname(args):
    name = args.model + f'-{args.patch_size}'
    name += f'_d{args.depth}w{args.width}h{args.heads}dh{args.dim_head}'
    if args.struct != 'none':
        name += f'_{args.struct}_{args.layers}_tt_{args.tt_dim}_{args.tt_rank}'
    name += f'_lr{args.lr}wd{args.weight_decay}bs{args.batch_size}'
    return name


def get_filename(args):
    return runname(args) + '.pth.tar'


# def save_checkpoint(state, is_best, filename='checkpoint.pth.tar'):
#    torch.save(state, filename)
#    if is_best:
#        shutil.copyfile(filename, 'model_best.pth.tar')


def save_checkpoint(state, save_dir=None, filename='checkpoint.pth.tar'):
    if save_dir is not None:
        if not os.path.isdir(save_dir):
            os.makedirs(save_dir, )
    torch.save(state, os.path.join(save_dir, filename))


class Summary(Enum):
    NONE = 0
    AVERAGE = 1
    SUM = 2
    COUNT = 3


class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f', summary_type=Summary.AVERAGE):
        self.name = name
        self.fmt = fmt
        self.summary_type = summary_type
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def all_reduce(self):
        if torch.cuda.is_available():
            device = torch.device("cuda")
        elif torch.backends.mps.is_available():
            device = torch.device("mps")
        else:
            device = torch.device("cpu")
        total = torch.tensor([self.sum, self.count], dtype=torch.float32, device=device)
        dist.all_reduce(total, dist.ReduceOp.SUM, async_op=False)
        self.sum, self.count = total.tolist()
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)

    def summary(self):
        fmtstr = ''
        if self.summary_type is Summary.NONE:
            fmtstr = ''
        elif self.summary_type is Summary.AVERAGE:
            fmtstr = '{name} {avg:.3f}'
        elif self.summary_type is Summary.SUM:
            fmtstr = '{name} {sum:.3f}'
        elif self.summary_type is Summary.COUNT:
            fmtstr = '{name} {count:.3f}'
        else:
            raise ValueError('invalid summary type %r' % self.summary_type)

        return fmtstr.format(**self.__dict__)


class ProgressMeter(object):
    def __init__(self, num_batches, meters, prefix=""):
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch):
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        print('\t'.join(entries))

    def display_summary(self):
        entries = [" *"]
        entries += [meter.summary() for meter in self.meters]
        print(' '.join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = '{:' + str(num_digits) + 'd}'
        return '[' + fmt + '/' + fmt.format(num_batches) + ']'


def accuracy(output, target, topk=(1, )):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res


if __name__ == '__main__':
    main()
