from tempfile import NamedTemporaryFile
from contextlib import contextmanager

from multiprocessing import Pipe, Process, TimeoutError

from typing import Callable, Optional, Tuple, Any, Dict


@contextmanager
def temppath(binary: bytes):
    fp = NamedTemporaryFile(mode="w+b")
    binary_path = fp.name
    fp.write(binary)
    try:
        yield binary_path
    finally:
        fp.close()

        
def run_in_process(fn: Callable, timeout: Optional[float] = None, args: Tuple[Any] = (), 
                   kwargs: Dict[str, Any] = {}):
    """Runs a callable in a separate process
    
    Args:
        fn: Callable to be executed.
        timeout: Time to wait before terminating the process. If None, the process will not be terminated until the 
            result is returned.
        args: Arguments to pass to `fn`.
        kwargs: Keyword arguments to pass to `fn`.
    
    Raises:
        TimeoutError: if the process is terminated before returning the result.
    
    Returns:
        The result `fn(*args, **kwargs)`.
    """
    front, back = Pipe(False)
    
    def target(*args, **kwargs):
        result = fn(*args, **kwargs)
        back.send(result)
    
    result = None
    p = Process(target=target, args=args, kwargs=kwargs)
    p.start()
    ready = front.poll(timeout)
    if ready:
        result = front.recv()
    else:
        p.terminate()
        raise TimeoutError(f"Failed to complete in {timeout} seconds")
    
    return result


def clamp(n: float, smallest: float, largest: float) -> float: 
    """Clamp an integer to a specified range
    
    Returns:
        Number in the interval [smallest, largest]
    """
    return max(smallest, min(n, largest))