import json
import time
import base64
import requests
from concurrent.futures import ThreadPoolExecutor
from tqdm import tqdm
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

def create_retry_session(retries=5, backoff_factor=1):
    session = requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=[500, 502, 503, 504],
        allowed_methods=frozenset(['POST'])
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session

def image_to_base64(image_path):
    with open(image_path, "rb") as image_file:
        return base64.b64encode(image_file.read()).decode('utf-8')

def enhance_instruction(data_entry, max_retries=3, retry_delay=1):
    human_instruction = data_entry["conversations"][0]["value"]
    image_url = data_entry["images"][-1]

    messages = [
        {
            "role": "user",
            "content": [
                {
                    "type": "text",
                    "text": f"""You are an AI assistant specialized in instruction obfuscation. Your task is to transform simple instructions into challenging formulations while preserving their core functional intent. Follow these precise guidelines:

1. **Contextual Embedding**  
   Replace direct object references with environmental cues (e.g., "the utensil adjacent to the cutting board" instead of "knife") and functional descriptors ("the liquid dispenser near the sink" instead of "soap bottle").

2. **Strategic Redundancy**  
   Introduce plausible environmental details that create cognitive noise without altering task requirements.

3. **Progressive Specification**  
   Gradually narrow down object identification through layered descriptions:  
   - General category → Spatial relationship → Functional implication  
   (Example: "the elongated tool" → "beside the stove" → "used for stirring liquids")

**Constraints:**  
- Maintain strict one-to-one action mapping with the original instruction  
- Base all descriptions exclusively on observable visual elements  
- Avoid technical terms or unnatural phrasing  
- Apply only one or two suitable advancements at one time
- Use natural, concise language without listing attributes

Original: {human_instruction}  
Cognitively Enhanced Version: """
                },
                {
                    "type": "image_url",
                    "image_url": {
                        "url": f"data:image/png;base64,{image_to_base64(image_url)}"
                    }
                }
            ]
        }
    ]

    session = create_retry_session()
    
    for attempt in range(max_retries + 1):
        try:
            response = session.post(
                "http://0.0.0.0:8000/v1/chat/completions",
                headers={
                    "Content-Type": "application/json",
                    "Authorization": f"Bearer sk-111"
                },
                json={
                    "model": "qwen2vl",
                    "messages": messages,
                    "max_tokens": 300,
                    "temperature": 0.7,
                    "top_p": 0.9
                },
                timeout=30
            )
            response.raise_for_status()  \
            return response.json()["choices"][0]["message"]["content"].strip()
            
        except Exception as e:
            if attempt < max_retries:
                wait_time = retry_delay * (2 ** attempt) 
                time.sleep(wait_time)
            else:
                raise Exception(f"Error after {max_retries} times: {str(e)}")

def process_entry(entry):
    try:
        enhanced = enhance_instruction(entry)
        return {
            "original": entry["conversations"][0]["value"],
            "enhanced": enhanced,
            "image": entry["images"][-1]
        }
    except Exception as e:
        print(f"最终处理失败: {str(e)}")
        return None

json_list = ["/path/to/data.json"]

def process_entry(entry):
    try:
        enhanced = enhance_instruction(entry)
        return {
            "original": entry["conversations"][0]["value"],
            "enhanced": enhanced,
            "image": entry["images"][-1]
        }
    except Exception as e:
        print(f"处理失败: {str(e)}")
        return None

for path in json_list:
    name = path.split("/")[-1].split(".")[0]
    with open(path) as f:
        dataset = json.load(f)

    enhanced_data = []
    failed_count = 0

    with ThreadPoolExecutor(max_workers=100) as executor:  
        futures = [executor.submit(process_entry, entry) for entry in dataset]
        
        progress = tqdm(total=len(futures), desc=f"Processing {name}", ncols=100)
        for future in futures:
            result = future.result()
            if result:
                enhanced_data.append(result)
            else:
                failed_count += 1
            progress.update()
        progress.close()


    save_path = "path/to/save"

    with open(save_path, "w") as f:
        json.dump(enhanced_data, f, indent=2)
    print(f"Saved in {save_path}")



    