"""SAMPLING ONLY."""

import torch
import numpy as np

from ldm.modules.diffusionmodules.util import make_ddim_sampling_parameters, make_ddim_timesteps, noise_like, \
    extract_into_tensor


device = torch.device("cuda") if torch.cuda.is_available() else torch.device("cpu")

class DDIMSampler(object):
    def __init__(self, model, schedule="linear", **kwargs):
        super().__init__()
        self.model = model
        self.ddpm_num_timesteps = model.num_timesteps
        self.schedule = schedule

    def register_buffer(self, name, attr):
        if type(attr) == torch.Tensor:
            if attr.device != torch.device("cuda"):
                attr = attr.to(torch.device("cuda"))
        setattr(self, name, attr)

    def make_schedule(self, ddim_num_steps, ddim_num_steps_uncond, ddim_discretize="uniform", ddim_eta=0., verbose=True):
        self.ddim_timesteps = make_ddim_timesteps(ddim_discr_method=ddim_discretize, num_ddim_timesteps=ddim_num_steps_uncond,
                                                  num_ddpm_timesteps=self.ddpm_num_timesteps,verbose=verbose)
        self.ddim_timesteps_cond = make_ddim_timesteps(ddim_discr_method=ddim_discretize, num_ddim_timesteps=ddim_num_steps,
                                                  num_ddpm_timesteps=self.ddpm_num_timesteps,verbose=verbose)
        
        alphas_cumprod = self.model.alphas_cumprod
        assert alphas_cumprod.shape[0] == self.ddpm_num_timesteps, 'alphas have to be defined for each timestep'
        to_torch = lambda x: x.clone().detach().to(torch.float32).to(device)

        self.register_buffer('betas', to_torch(self.model.betas))
        self.register_buffer('alphas_cumprod', to_torch(alphas_cumprod))
        self.register_buffer('alphas_cumprod_prev', to_torch(self.model.alphas_cumprod_prev))

        # calculations for diffusion q(x_t | x_{t-1}) and others
        self.register_buffer('sqrt_alphas_cumprod', to_torch(np.sqrt(alphas_cumprod.cpu())))
        self.register_buffer('sqrt_one_minus_alphas_cumprod', to_torch(np.sqrt(1. - alphas_cumprod.cpu())))
        self.register_buffer('log_one_minus_alphas_cumprod', to_torch(np.log(1. - alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recip_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu())))
        self.register_buffer('sqrt_recipm1_alphas_cumprod', to_torch(np.sqrt(1. / alphas_cumprod.cpu() - 1)))

        # ddim sampling parameters
        ddim_sigmas, ddim_alphas, ddim_alphas_prev, ddim_sigmas_orig = make_ddim_sampling_parameters(alphacums=alphas_cumprod.cpu(),
                                                                                   ddim_timesteps=self.ddim_timesteps,
                                                                                   eta=ddim_eta,verbose=verbose)
        self.register_buffer('ddim_sigmas', ddim_sigmas)
        self.register_buffer('ddim_alphas', ddim_alphas)
        self.register_buffer('ddim_alphas_prev', ddim_alphas_prev)
        self.register_buffer('ddim_sqrt_one_minus_alphas', np.sqrt(1. - ddim_alphas))
        sigmas_for_original_sampling_steps = ddim_eta * torch.sqrt(
            (1 - self.alphas_cumprod_prev) / (1 - self.alphas_cumprod) * (
                        1 - self.alphas_cumprod / self.alphas_cumprod_prev))
        self.register_buffer('ddim_sigmas_for_original_num_steps', sigmas_for_original_sampling_steps)
        self.register_buffer('ddim_sigmas_orig', ddim_sigmas_orig)
        
        # ddim sampling parameters for conditional sampling
        ddim_sigmas_cond, ddim_alphas_cond, ddim_alphas_prev_cond, ddim_sigmas_orig_cond = make_ddim_sampling_parameters(alphacums=alphas_cumprod.cpu(),
                                                                                   ddim_timesteps=self.ddim_timesteps_cond,
                                                                                   eta=ddim_eta,verbose=verbose)
        self.register_buffer('ddim_sigmas_cond', ddim_sigmas_cond)
        self.register_buffer('ddim_alphas_cond', ddim_alphas_cond)
        self.register_buffer('ddim_alphas_prev_cond', ddim_alphas_prev_cond)
        self.register_buffer('ddim_sqrt_one_minus_alphas_cond', np.sqrt(1. - ddim_alphas_cond))
        self.register_buffer('ddim_sigmas_orig_cond', ddim_sigmas_orig_cond)

    @torch.no_grad()
    def sample(self,
               S,
               S_uncond,
               batch_size,
               shape,
               conditioning=None,
               callback=None,
               normals_sequence=None,
               img_callback=None,
               quantize_x0=False,
               eta=0.,
               mask=None,
               x0=None,
               temperature=1.,
               noise_dropout=0.,
               score_corrector=None,
               corrector_kwargs=None,
               verbose=True,
               x_T=None,
               log_every_t=100,
               unconditional_guidance_scale=1.,
               unconditional_conditioning=None,
               uncond_guide=0.5,
               coef_type='vanilla',
               layer_skip=False,
               n_layer=3,
               # this has to come in the same format as the conditioning, # e.g. as encoded tokens, ...
               **kwargs
               ):
        if conditioning is not None:
            if isinstance(conditioning, dict):
                cbs = conditioning[list(conditioning.keys())[0]].shape[0]
                if cbs != batch_size:
                    print(f"Warning: Got {cbs} conditionings but batch-size is {batch_size}")
            else:
                if conditioning.shape[0] != batch_size:
                    print(f"Warning: Got {conditioning.shape[0]} conditionings but batch-size is {batch_size}")

        self.make_schedule(ddim_num_steps=S, ddim_num_steps_uncond=S_uncond, ddim_eta=eta, verbose=verbose)
        # sampling
        C, H, W = shape
        size = (batch_size, C, H, W)

        samples, intermediates = self.ddim_sampling(conditioning, size,
                                                    callback=callback,
                                                    img_callback=img_callback,
                                                    quantize_denoised=quantize_x0,
                                                    mask=mask, x0=x0,
                                                    ddim_use_original_steps=False,
                                                    noise_dropout=noise_dropout,
                                                    temperature=temperature,
                                                    score_corrector=score_corrector,
                                                    corrector_kwargs=corrector_kwargs,
                                                    x_T=x_T,
                                                    log_every_t=log_every_t,
                                                    unconditional_guidance_scale=unconditional_guidance_scale,
                                                    unconditional_conditioning=unconditional_conditioning,
                                                    uncond_guide=uncond_guide,
                                                    coef_type=coef_type,
                                                    layer_skip=layer_skip,
                                                    n_layer=n_layer,
                                                    )
        return samples, intermediates

    @torch.no_grad()
    def ddim_sampling(self, cond, shape,
                      x_T=None, ddim_use_original_steps=False,
                      callback=None, timesteps=None, quantize_denoised=False,
                      mask=None, x0=None, img_callback=None, log_every_t=100,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None, uncond_guide=0.5, coef_type='vanilla',
                      layer_skip=False, n_layer=3):
        device = self.model.betas.device
        b = shape[0]
        if x_T is None:
            img = torch.randn(shape, device=device)
        else:
            img = x_T
        img_g = img.clone()

        if timesteps is None:
            timesteps = self.ddpm_num_timesteps if ddim_use_original_steps else self.ddim_timesteps
        elif timesteps is not None and not ddim_use_original_steps:
            subset_end = int(min(timesteps / self.ddim_timesteps.shape[0], 1) * self.ddim_timesteps.shape[0]) - 1
            timesteps = self.ddim_timesteps[:subset_end]

        intermediates = {}
        time_range = reversed(range(0,timesteps)) if ddim_use_original_steps else np.flip(timesteps)
        time_range_cond = np.flip(self.ddim_timesteps_cond)
        total_steps = timesteps if ddim_use_original_steps else timesteps.shape[0]


        iterator = time_range
        decrypt_unconditional_conditioning = unconditional_conditioning

        for module in self.model.model.diffusion_model.modules():
            module.fhe_mode = False

        img_p = img_g.clone()
        fix_noise = noise_like(img_g.shape, device, False)
        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((b,), step, device=device, dtype=torch.long)

            outs_g = self.p_sample_ddim_g(img_g, decrypt_unconditional_conditioning, ts, index=index, use_original_steps=ddim_use_original_steps,
                                      fix_noise=fix_noise, quantize_denoised=quantize_denoised, temperature=temperature,
                                      noise_dropout=noise_dropout, score_corrector=score_corrector,
                                      corrector_kwargs=corrector_kwargs,
                                      unconditional_guidance_scale=unconditional_guidance_scale,
                                      unconditional_conditioning=decrypt_unconditional_conditioning)
            img_g, _ = outs_g
            
            if step in time_range_cond:
                step_idx = np.arange(len(time_range_cond))[time_range_cond == step][0]
                index_cond = len(time_range_cond) - step_idx - 1
                ts_cond = torch.full((b,), step, device=device, dtype=torch.long)
                
                alphas = self.ddim_alphas_cond
                alphas_prev = self.ddim_alphas_prev_cond
                sqrt_one_minus_alphas = self.ddim_sqrt_one_minus_alphas_cond
                sigmas = self.ddim_sigmas_cond
                
                a_t = torch.full((b, 1, 1, 1), alphas[index_cond], device=device)
                a_prev = torch.full((b, 1, 1, 1), alphas_prev[index_cond], device=device)
                sigma_t = torch.full((b, 1, 1, 1), sigmas[index_cond], device=device)
                sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index_cond],device=device)
                
                denominator = ((1. - a_prev - sigma_t**2).sqrt() - a_prev.sqrt() * sqrt_one_minus_at / a_t.sqrt())
                e_t_ug = (img_g - (a_prev.sqrt() * img_p / a_t.sqrt()) - sigma_t * fix_noise * temperature) / denominator
                
                img_p = img_g.clone()  
                
                for module in self.model.model.diffusion_model.modules():
                    module.fhe_mode = True

                outs = self.p_sample_ddim(img, e_t_ug, cond, ts_cond, index=index_cond, use_original_steps=ddim_use_original_steps,
                                      fix_noise=fix_noise, quantize_denoised=quantize_denoised, temperature=temperature,
                                      noise_dropout=noise_dropout, score_corrector=score_corrector,
                                      corrector_kwargs=corrector_kwargs,
                                      unconditional_guidance_scale=unconditional_guidance_scale,
                                      unconditional_conditioning=unconditional_conditioning, uncond_guide=uncond_guide, coef_type=coef_type)
                img, _ = outs

                for module in self.model.model.diffusion_model.modules():
                    module.fhe_mode = False

                fix_noise = noise_like(img_g.shape, device, False)

        return img, intermediates

    @torch.no_grad()
    def p_sample_ddim(self, x, e_t_ug, c, t, index, repeat_noise=False, use_original_steps=False, fix_noise=0, quantize_denoised=False,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None, uncond_guide=0.5, coef_type='vanilla',
                      ):
        b, *_, device = *x.shape, x.device

        alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas_cond
        alphas_prev = self.model.alphas_cumprod_prev if use_original_steps else self.ddim_alphas_prev_cond
        sqrt_one_minus_alphas = self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas_cond
        sigmas = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas_cond
        sigmas_orig = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas_orig_cond
        
        if coef_type == 'lambda':
            coef = torch.full((b, 1, 1, 1), sigmas_orig[index]/alphas[index], device=device)
        elif coef_type == 'sigma_lambda':
            coef = torch.full((b, 1, 1, 1), (sigmas_orig[index]**2)/alphas[index], device=device)
        elif coef_type == 'square_lambda':
            coef = torch.full((b, 1, 1, 1), (sigmas_orig[index]/alphas[index])**2, device=device)
        elif coef_type == 'vanilla':
            coef = torch.full((b, 1, 1, 1), 1, device=device)
        else:
            coef = None
        
        if unconditional_conditioning is None or unconditional_guidance_scale == 1.:
            e_t = self.model.apply_model(x, t, c)
        else:
            x_in = torch.cat([x] * 2)
            t_in = torch.cat([t] * 2)
            c_in = torch.cat([unconditional_conditioning, c])

            output = self.model.apply_model(x_in, t_in, c_in)
            n_ch = output.shape[0] // 2
            e_t_uncond, e_t = output[:n_ch], output[n_ch:]
            
            if e_t_ug is None:
                e_t_ug = e_t

            e_t = e_t_uncond + unconditional_guidance_scale * (e_t - e_t_uncond) - uncond_guide * coef * (e_t_ug - e_t)

        if score_corrector is not None:
            assert self.model.parameterization == "eps"
            e_t = score_corrector.modify_score(self.model, e_t, x, t, c, **corrector_kwargs)

        # select parameters corresponding to the currently considered timestep
        a_t = torch.full((b, 1, 1, 1), alphas[index], device=device)
        a_prev = torch.full((b, 1, 1, 1), alphas_prev[index], device=device)
        sigma_t = torch.full((b, 1, 1, 1), sigmas[index], device=device)
        sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index],device=device)
        
        # current prediction for x_0
        pred_x0 = (x - sqrt_one_minus_at * e_t) / a_t.sqrt()
        if quantize_denoised:
            pred_x0, _, *_ = self.model.first_stage_model.quantize(pred_x0)
        # direction pointing to x_t
        dir_xt = (1. - a_prev - sigma_t**2).sqrt() * e_t
        noise = sigma_t * fix_noise * temperature
        if noise_dropout > 0.:
            noise = torch.nn.functional.dropout(noise, p=noise_dropout)
        x_prev = a_prev.sqrt() * pred_x0 + dir_xt + noise
        return x_prev, pred_x0
    

    @torch.no_grad()
    def p_sample_ddim_g(self, x, c, t, index, repeat_noise=False, use_original_steps=False, fix_noise=0, quantize_denoised=False,
                      temperature=1., noise_dropout=0., score_corrector=None, corrector_kwargs=None,
                      unconditional_guidance_scale=1., unconditional_conditioning=None):
        b, *_, device = *x.shape, x.device

        if unconditional_conditioning is None or unconditional_guidance_scale == 1.:
            e_t = self.model.apply_model(x, t, c)
        else:
            x_in = x
            t_in = t
            c_in = unconditional_conditioning
            e_t = self.model.apply_model(x_in, t_in, c_in)

        if score_corrector is not None:
            assert self.model.parameterization == "eps"
            e_t = score_corrector.modify_score(self.model, e_t, x, t, c, **corrector_kwargs)

        alphas = self.model.alphas_cumprod if use_original_steps else self.ddim_alphas
        alphas_prev = self.model.alphas_cumprod_prev if use_original_steps else self.ddim_alphas_prev
        sqrt_one_minus_alphas = self.model.sqrt_one_minus_alphas_cumprod if use_original_steps else self.ddim_sqrt_one_minus_alphas
        sigmas = self.model.ddim_sigmas_for_original_num_steps if use_original_steps else self.ddim_sigmas
        # select parameters corresponding to the currently considered timestep
        a_t = torch.full((b, 1, 1, 1), alphas[index], device=device)
        a_prev = torch.full((b, 1, 1, 1), alphas_prev[index], device=device)
        sigma_t = torch.full((b, 1, 1, 1), sigmas[index], device=device)
        sqrt_one_minus_at = torch.full((b, 1, 1, 1), sqrt_one_minus_alphas[index],device=device)

        # current prediction for x_0
        pred_x0 = (x - sqrt_one_minus_at * e_t) / a_t.sqrt()
        if quantize_denoised:
            pred_x0, _, *_ = self.model.first_stage_model.quantize(pred_x0)
        # direction pointing to x_t
        dir_xt = (1. - a_prev - sigma_t**2).sqrt() * e_t
        noise = sigma_t * fix_noise * temperature
        if noise_dropout > 0.:
            noise = torch.nn.functional.dropout(noise, p=noise_dropout)
        x_prev = a_prev.sqrt() * pred_x0 + dir_xt + noise
        return x_prev, pred_x0

    @torch.no_grad()
    def stochastic_encode(self, x0, t, use_original_steps=False, noise=None):
        # fast, but does not allow for exact reconstruction
        # t serves as an index to gather the correct alphas
        if use_original_steps:
            sqrt_alphas_cumprod = self.sqrt_alphas_cumprod
            sqrt_one_minus_alphas_cumprod = self.sqrt_one_minus_alphas_cumprod
        else:
            sqrt_alphas_cumprod = torch.sqrt(self.ddim_alphas)
            sqrt_one_minus_alphas_cumprod = self.ddim_sqrt_one_minus_alphas

        if noise is None:
            noise = torch.randn_like(x0)
        return (extract_into_tensor(sqrt_alphas_cumprod, t, x0.shape) * x0 +
                extract_into_tensor(sqrt_one_minus_alphas_cumprod, t, x0.shape) * noise)

    @torch.no_grad()
    def decode(self, x_latent, cond, t_start, unconditional_guidance_scale=1.0, unconditional_conditioning=None,
               use_original_steps=False):

        timesteps = np.arange(self.ddpm_num_timesteps) if use_original_steps else self.ddim_timesteps
        timesteps = timesteps[:t_start]

        time_range = np.flip(timesteps)
        total_steps = timesteps.shape[0]
        
        iterator = time_range
        x_dec = x_latent
        for i, step in enumerate(iterator):
            index = total_steps - i - 1
            ts = torch.full((x_latent.shape[0],), step, device=x_latent.device, dtype=torch.long)
            x_dec, _ = self.p_sample_ddim(x_dec, cond, ts, index=index, use_original_steps=use_original_steps,
                                          unconditional_guidance_scale=unconditional_guidance_scale,
                                          unconditional_conditioning=unconditional_conditioning)
        return x_dec