import re
import json
import ast

def extract_last_code_block(text: str) -> str:
    """
    Extract the last code block from the given text.
    
    Args:
        text (str): The input text containing code blocks
        
    Returns:
        str: The content of the last code block, or None if no code block is found
    """
    code_blocks = re.findall(r'```.*?\n(.*?)```', text, re.DOTALL)
    if not code_blocks:
        code_blocks = re.findall(r'```(.*?)```', text, re.DOTALL)
    return code_blocks[-1].strip() if code_blocks else None

def verify(pred: str, answer: str, meta: dict) -> int:
    """
    Verify if the model's prediction matches the correct answer for the 24 game.
    
    Args:
        pred (str): The model's prediction
        answer (str): The correct answer
        meta (dict): Additional metadata containing the input numbers
        
    Returns:
        int: 1 if the prediction is correct, 0 otherwise
    """
    if isinstance(answer, str):
        try:
            answer = json.loads(answer)
        except json.JSONDecodeError:
            pass
    
    if isinstance(meta, str):
        meta = json.loads(meta)
    elif not isinstance(meta, dict):
        raise ValueError('meta should be dict or str')
        
    input_numbers = meta["question"]
    num_len = len(input_numbers)
    final_answer = extract_last_code_block(pred)
    
    if final_answer is None or not isinstance(final_answer, str):
        return 0
        
    if "cannot form 24" in answer or '无法组成' in answer:
        if "cannot form 24" in final_answer or '无法组成' in final_answer:
            return 1
        return 0
        
    expression = final_answer
    try:
        expression = expression.replace("x", "*")
        expression = expression.replace("×", "*")
        expression = expression.replace("÷", "/")
        
        if abs(eval(expression) - 24) > 1e-4:
            return 0
            
        number_pattern = r"\d+"
        extract_numbers = re.findall(number_pattern, expression)
        if len(extract_numbers) != num_len:
            return 0
            
        sorted_extract_numbers = sorted(extract_numbers)
        original_numbers = [str(num) for num in input_numbers]
        sorted_original_numbers = sorted(original_numbers)
        
        for i in range(num_len):
            if sorted_extract_numbers[i] != sorted_original_numbers[i]:
                return 0
        return 1
    except Exception:
        return 0