"""
  _emp_cov_quadform(A::Array{Float64, 3}, x::Vector{Float64})

Implement the weighted quadratic form

\\[
  (1 / n) ∑_i x'A_{:, i, :} A_{:, i, :}^T x
\\]

efficiently for a given tensor `A` and input `x`.
"""
function _emp_cov_quadform(A::Array{Float64, 3}, x::Vector{Float64})
  m = size(A, 2)
  z = zeros(m)
  @inbounds for i = 1:m
    z[i] = (1 / m) * LinearAlgebra.norm(A[:, i, :] * x)^2
  end
  return sum(z)
end

"""
  _emp_cov_scores(A::Array{Float64, 3}, evec::Vector{Float64})

Compute the outlier scores given a tensor `A` holding centered samples along
its 2nd dimension and the eigenvector `evec` of the empirical covariance matrix
formed from `A`.

Returns the vector of outlier scores.
"""
function _emp_cov_scores(A::Array{Float64, 3}, evec::Vector{Float64})
  m = size(A, 2)
  z = zeros(m)
  @inbounds for i = 1:m
    z[i] = LinearAlgebra.norm(A[:, i, :]' * evec)^2
  end
  return z
end

"""
  _emp_cov_op(A::Array{Float64, 3}, x::Vector{Float64})

Implement the operation `x ↦ (1 / m) * ∑_i A[:, i, :] * A[:, i, :]' x`
efficiently for a given tensor `A` and input `x`.

Returns the resulting vector.
"""
function _emp_cov_op(A::Array{Float64, 3}, x::Vector{Float64})
  _, m, d = size(A)
  z = zeros(d)
  @inbounds for i = 1:m
    z += (1 / m) * A[:, i, :]' * (A[:, i, :] * x)
  end
  return z
end
