import argparse
import nasspace
import datasets
import random
import numpy as np
import torch
import os
from scores import get_score_func
from sklearn.metrics.pairwise import cosine_similarity


parser = argparse.ArgumentParser(description='NAS Without Training')
parser.add_argument('--data_loc', default='../fishersearch_randomwirenetworks/cifardata/', type=str, help='dataset folder')
parser.add_argument('--api_loc', default='../fimflam/NAS-Bench-201-v1_0-e61699.pth',
                    type=str, help='path to API')
parser.add_argument('--save_loc', default='results', type=str, help='folder to save results')
parser.add_argument('--save_string', default='naswot', type=str, help='prefix of results file')
parser.add_argument('--nasspace', default='nasbench201', type=str, help='the nas search space to use')
parser.add_argument('--batch_size', default=256, type=int)
parser.add_argument('--repeat', default=256, type=int, help='how often to repeat a single image with a batch')
parser.add_argument('--augtype', default='gaussnoise', type=str, help='which perturbations to use')
parser.add_argument('--sigma', default=0.001, type=float, help='noise level if augtype is "gaussnoise"')
parser.add_argument('--GPU', default='0', type=str)
parser.add_argument('--seed', default=1, type=int)
parser.add_argument('--trainval', action='store_true')
parser.add_argument('--activations', action='store_true')
parser.add_argument('--cosine', action='store_true')
parser.add_argument('--dataset', default='cifar10', type=str)
parser.add_argument('--n_samples', default=100, type=int)
parser.add_argument('--n_runs', default=500, type=int)
parser.add_argument('--stem_out_channels', default=16, type=int, help='output channels of stem convolution (nasbench101)')
parser.add_argument('--num_stacks', default=3, type=int, help='#stacks of modules (nasbench101)')
parser.add_argument('--num_modules_per_stack', default=3, type=int, help='#modules per stack (nasbench101)')
parser.add_argument('--num_labels', default=1, type=int, help='#classes (nasbench101)')

args = parser.parse_args()
os.environ['CUDA_VISIBLE_DEVICES'] = args.GPU

# Reproducibility
torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmark = False
random.seed(args.seed)
np.random.seed(args.seed)
torch.manual_seed(args.seed)


def get_batch_jacobian(net, x, target, device, args=None):
    net.zero_grad()
    x.requires_grad_(True)
    y, ints = net(x)
    y.backward(torch.ones_like(y))
    jacob = x.grad.detach()
    return jacob, target.detach(), ints.detach()

device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
searchspace = nasspace.get_search_space(args)
train_loader = datasets.get_data(args.dataset, args.data_loc, args.trainval, args.batch_size, args.augtype, args.repeat, args)
os.makedirs(args.save_loc, exist_ok=True)

filename = f'{args.save_loc}/{args.save_string}_correlationmatrix_{args.activations}_{args.nasspace}_{args.dataset}_{args.augtype}_{args.sigma}_{args.repeat}_{args.trainval}_{args.batch_size}_{args.seed}{"_cosine" if args.cosine else ""}'
accsfilename = f'{args.save_loc}/{args.save_string}_correlationmatrixaccs_{args.activations}_{args.nasspace}_{args.dataset}_{args.augtype}_{args.sigma}_{args.repeat}_{args.trainval}_{args.batch_size}_{args.seed}{"_cosine" if args.cosine else ""}'

if args.dataset == 'cifar10':
    acc_type = 'ori-test'
    val_acc_type = 'x-valid'
else:
    acc_type = 'x-test'
    val_acc_type = 'x-valid'
#scores = np.zeros((len(searchspace), args.batch_size*(args.batch_size-1)//2), dtype=np.float32)
num_networks = 110000
if args.activations:
    num_activations = searchspace.num_activations()
    print(f'num_activations {num_activations}')
    scores = np.zeros((num_networks, args.batch_size*num_activations), dtype=np.float32)
else:
    scores = np.zeros((num_networks, args.batch_size*(args.batch_size-1)//2), dtype=np.float32)
accs = np.zeros(num_networks, dtype=np.float32)
for i, ((uid, network), j) in enumerate(zip(searchspace, range(num_networks))):
    # Reproducibility
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    data_iterator = iter(train_loader)
    x, target = next(data_iterator)
    x, target = x.to(device), target.to(device)
    network = network.to(device)
    jacobs, labels, ints = get_batch_jacobian(network, x, target, device, args)
    if args.activations:
        scores[i, :] = ints.cpu().numpy().flatten()
    else:
        jacobs = jacobs.detach().reshape(jacobs.size(0), -1).cpu().numpy()
        if args.cosine:
            corrs = cosine_similarity(jacobs)
        else:
            corrs = np.corrcoef(jacobs)
        corrs = corrs.astype(np.float32)
        scores[i, :] = corrs[np.tril_indices(corrs.shape[0], -1)].flatten()
    accs[i] = searchspace.get_accuracy(uid, acc_type, args.trainval)
    if i % 1000 == 0:
        np.save(filename, scores)
        np.save(accsfilename, accs)
np.save(filename, scores)
np.save(accsfilename, accs)

