import torch
import torch.nn as nn
import torch.nn.functional as F


def conv3x3(in_channels, out_channels, stride=1):
    return nn.Conv2d(in_channels, out_channels, kernel_size=3, stride=stride,
                     padding=1, bias=False)


class PreActBottleneck(nn.Module):
    """
    Pre-activation version of the original Bottleneck module.
    """
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(PreActBottleneck, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)

        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False)
            )

    def forward(self, x):
        out = F.relu(self.bn1(x))
        shortcut = self.shortcut(out) if hasattr(self, 'shortcut') else x
        out = self.conv1(out)
        out = self.conv2(F.relu(self.bn2(out)))
        out = self.conv3(F.relu(self.bn3(out)))
        out += shortcut
        return out


class ResNetV2CIFAR(nn.Module):
    def __init__(self, depth, num_classes, init='he'):
        super(ResNetV2CIFAR, self).__init__()
        block = PreActBottleneck
        multiplier = (depth - 2) // 9
        filters = [16, 16, 32, 64]

        self.conv1 = conv3x3(3, filters[0])
        self.in_channels = filters[0]
        self.block1 = self._make_layers(block, multiplier, filters[1], 1)
        self.block2 = self._make_layers(block, multiplier, filters[2], 2)
        self.block3 = self._make_layers(block, multiplier, filters[3], 2)
        self.bn1 = nn.BatchNorm2d(self.in_channels)
        self.relu = nn.ReLU(inplace=True)
        self.gp = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(self.in_channels, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.ones_(m.weight)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)

    def _make_layers(self, block, num_block, out_channels, stride):
        layers = []
        layers.append(block(self.in_channels, out_channels, stride))
        self.in_channels = block.expansion * out_channels
        for i in range(num_block - 1):
            layers.append(block(self.in_channels, out_channels))
            self.in_channels = block.expansion * out_channels
        return nn.Sequential(*layers)

    def forward(self, x):
        conv1 = self.conv1(x)
        block1 = self.block1(conv1)
        block2 = self.block2(block1)
        block3 = self.block3(block2)
        before_gp = self.relu(self.bn1(block3))
        after_gp_fm = self.gp(before_gp)
        after_gp_vec = after_gp_fm.view(after_gp_fm.size(0), -1)

        features = dict()
        features['conv1'] = conv1
        features['block1'] = block1
        features['block2'] = block2
        features['block3'] = block3
        features['before_gp'] = before_gp
        features['after_gp_vec'] = after_gp_vec

        return self.fc(after_gp_vec), features


class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, dropRate=0.0):
        super(BasicBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.relu1 = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1,
                               padding=1, bias=False)
        self.droprate = dropRate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride,
                                                                padding=0, bias=False) or None

    def forward(self, x):
        if not self.equalInOut:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.droprate > 0:
            out = F.dropout(out, p=self.droprate, training=self.training)
        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)


class NetworkBlock(nn.Module):

    def __init__(self, nb_layers, in_planes, out_planes, block, stride, dropRate=0.0):
        super(NetworkBlock, self).__init__()
        self.layer = self._make_layer(block, in_planes, out_planes, nb_layers, stride, dropRate)

    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, dropRate):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes, i == 0 and stride or 1, dropRate))
        return nn.Sequential(*layers)

    def forward(self, x):
        return self.layer(x)


class WideResNetCIFAR(nn.Module):

    def __init__(self, depth, num_classes, widen_factor=1, dropRate=0.0, init='he'):
        # Some popular setting: wrn-28-10, wrn-40-10.
        # wrn--4
        super(WideResNetCIFAR, self).__init__()
        nChannels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        assert((depth - 4) % 6 == 0)
        n = (depth - 4) // 6
        block = BasicBlock
        # 1st conv before any network block
        self.conv1 = nn.Conv2d(3, nChannels[0], kernel_size=3, stride=1,
                               padding=1, bias=False)
        # 1st block
        self.block1 = NetworkBlock(n, nChannels[0], nChannels[1], block, 1, dropRate)
        # 2nd block
        self.block2 = NetworkBlock(n, nChannels[1], nChannels[2], block, 2, dropRate)
        # 3rd block
        self.block3 = NetworkBlock(n, nChannels[2], nChannels[3], block, 2, dropRate)
        self.bn1 = nn.BatchNorm2d(nChannels[3])
        self.relu = nn.ReLU(inplace=True)
        # global average pooling and classifier
        self.gp = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(nChannels[3], num_classes)
        self.nChannels = nChannels[3]

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.ones_(m.weight)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)

    def forward(self, x):
        conv1 = self.conv1(x)
        block1 = self.block1(conv1)
        block2 = self.block2(block1)
        block3 = self.block3(block2)
        before_gp = self.relu(self.bn1(block3))
        after_gp_fm = self.gp(before_gp)
        after_gp_vec = after_gp_fm.view(-1, self.nChannels)

        features = dict()
        features['conv1'] = conv1
        features['block1'] = block1
        features['block2'] = block2
        features['block3'] = block3
        features['before_gp'] = before_gp
        features['after_gp_vec'] = after_gp_vec

        return self.fc(after_gp_vec), features


# lenet
class LeNet(nn.Module):
    def __init__(self, num_classes, init='he'):
        super(LeNet, self).__init__()
        self.conv1 = nn.Conv2d(3, 6, 5)
        self.pool = nn.MaxPool2d(2, 2)
        self.conv2 = nn.Conv2d(6, 16, 5)
        self.fc1 = nn.Linear(16 * 5 * 5, 120)
        self.fc2 = nn.Linear(120, 84)
        self.fc3 = nn.Linear(84, num_classes)
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.ones_(m.weight)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                if init == 'he':
                    nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                elif init == 'xavier':
                    # by default, tensorflow uses uniform with gain 1.0.
                    nn.init.xavier_uniform_(m.weight)
                else:
                    raise ValueError('Unknown initializer: %s' % init)
                if m.bias is not None:
                    nn.init.zeros_(m.bias)

    def forward(self, x):
        features = dict()
        x = self.pool(F.relu(self.conv1(x)))
        features['conv1'] = x
        x = self.pool(F.relu(self.conv2(x)))
        features['conv2'] = x
        x = x.view(-1, 16 * 5 * 5)
        x = F.relu(self.fc1(x))
        features['fc1'] = x
        x = F.relu(self.fc2(x))
        features['fc2'] = x
        x = self.fc3(x)
        return x, features


if __name__ == '__main__':
    net = ResNetV2CIFAR(56, 10)
    assert isinstance(net, torch.nn.Module)
    print(net)

    total_params = sum([p.data.nelement() for p in net.parameters()])
    print("Total number of params", total_params)
    print("Total layers", len(list(filter(lambda p: p.requires_grad and len(p.data.size())>1, net.parameters()))))
