import time
import torch


class Timestats(object):
    def __init__(self):
        self.start = self.last = self.enter = 0
        self.reset()

    def reset(self):
        self.start = self.last = time.time()

    def mark(self):
        now = time.time()
        interval = now - self.last
        self.last = now
        return interval

    def elapse(self):
        return time.time() - self.start

    def __enter__(self):
        self.enter = time.time()

    def __exit__(self, exc_type, exc_val, exc_tb):
        print(f"Time elapsed: {time.time() - self.enter}")


class Timer(object):
    def __init__(self, format_str='%d:%02d:%02d'):
        """
        Args:
            format_str: format of hour-minute-second
        """
        self.format_str = format_str
        self._start = time.time()
        self._last = self._start

    def reset(self):
        """
        Reset timer.
        """
        self._start = time.time()
        self._last = self._start

    def tick(self):
        '''
        Get time elapsed from lass tick.

        Returns:
            a formatted time string
        '''
        elapse = time.time() - self._last
        self._last = time.time()
        return self._elapse_str(elapse)

    def tock(self):
        '''
        Get time elapsed from start or last reset.

        Returns:
            a formatted time string
        '''
        elapse = time.time() - self._start
        return self._elapse_str(elapse)

    def _elapse_str(self, elapse):
        second = int(elapse % 60)
        minute = int((elapse // 60) % 60)
        hour = int(elapse // 3600)
        elapse_str = self.format_str % (hour, minute, second)
        return elapse_str


def init_seeds(seed=0):
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.cuda.manual_seed_all(seed)
    torch.backends.cudnn.deterministic = False
    torch.backends.cudnn.benchmark = True
