import torch

from extensions.rl_poisoneddoors.poisoneddoors_experiments.base import (
    PoisonedDoorsBaseExperimentConfig,
)
from utils.experiment_utils import PipelineStage, LinearDecay


class PoisonedDoorsBCThenPPOExperimentConfig(PoisonedDoorsBaseExperimentConfig):
    """Tackle (currently) the PoisonedDoors environment.

    Training with behavior cloning and then PPO.
    """

    GPU_ID = 1 if torch.cuda.is_available() else None
    USE_EXPERT = True

    @classmethod
    def extra_tag(cls):
        return "BCTeacherForcingThenPPO__bc_{}__lr_{}".format(cls.tf_ratio(), cls.lr())

    @classmethod
    def training_pipeline(cls, **kwargs):
        training_steps = cls.TOTAL_TRAIN_STEPS
        steps_imitation_stage = int(training_steps * cls.tf_ratio())
        steps_ppo_stage = training_steps - steps_imitation_stage

        ppo_info = cls.rl_loss_default("ppo", steps=steps_ppo_stage)
        imitation_info = cls.rl_loss_default("imitation")

        return cls._training_pipeline(
            named_losses={
                "imitation_loss": imitation_info["loss"],
                "ppo_loss": ppo_info["loss"],
            },
            pipeline_stages=[
                PipelineStage(
                    loss_names=["imitation_loss"],
                    max_stage_steps=training_steps,
                    teacher_forcing=LinearDecay(
                        startp=1.0, endp=1.0, steps=training_steps,
                    ),
                ),
                PipelineStage(
                    loss_names=["ppo_loss"],
                    max_stage_steps=steps_ppo_stage,
                    early_stopping_criterion=cls.task_info().get(
                        "early_stopping_criterion"
                    ),
                ),
            ],
            num_mini_batch=min(
                info["num_mini_batch"] for info in [ppo_info, imitation_info]
            ),
            update_repeats=min(
                info["update_repeats"] for info in [ppo_info, imitation_info]
            ),
        )
