import numpy as np
import matplotlib as mpl
import matplotlib.pyplot as plt
import os
import re
from collections import OrderedDict

def linestyle2dashes(style):
    mapping = {
        'solid': (0, ()),
        'dotted': (0, (1, 1)),
        'loosely dotted': (0, (1, 10)),
        'densely dotted': (0, (1, 1)),
        'dashed': (0, (5, 5)),
        'loosely dashed': (0, (5, 10)),
        'densely dashed': (0, (5, 1)),
        'dashdotted': (0, (3, 5, 1, 5)),
        'loosely dashdotted': (0, (3, 10, 1, 10)),
        'densely dashdotted': (0, (3, 1, 1, 1)),
        'dashdotdotted': (0, (3, 5, 1, 5, 1, 5)),
        'loosely dashdotdotted': (0, (3, 10, 1, 10, 1, 10)),
        'densely dashdotdotted': (0, (3, 1, 1, 1, 1, 1))
    }
    return mapping.get(style, (0, ()))


def plot_results(filename, environments, algorithms):
    """
    horizon: total rounds
    filename: base directory under './Results'
    environments: list of tuples (env_name, d) or list of env_name strings
    algorithms: list of algorithm name strings
    """
    # Styling
    mpl.rcParams.update({
        "axes.linewidth":0.75,
        "grid.linewidth":0.75,
        "lines.linewidth":1.5,
        "patch.linewidth":1.5,
        "xtick.major.size":3,
        "ytick.major.size":3,
        "pdf.fonttype":42,
        "ps.fonttype":42,
        "font.size":14,
        "axes.titlesize":"large",
        "axes.labelsize":"medium",
        "xtick.labelsize":"medium",
        "ytick.labelsize":"medium",
        "legend.fontsize":"large",
        "text.usetex":False
    })

    alg_map = {
        "e-greedy":   ["orange","solid", r"$\varepsilon$-Greedy"],
        "greedy":     ["orange","dashed","Greedy"],
        "LinUCB":     ["green","solid","LinUCB"],
        "LinTS":      ["black","solid","LinTS"],
        "FGTS":       ["black","dashed","FGTS"],
        "LinPHE":     ["purple","solid","LinPHE"],
        "RandLinUCB": ["red","solid","RandLinUCB"],
        "LinFP":      ["blue","solid","LinFP"],
        "FGFP":       ["blue","dashed","FGFP"],
        "OPAS-FGP":   ["purple","dashed","OPAS-FGP"],

        "log_e-greedy":   ["orange","solid", r"$\varepsilon$-Greedy"],
        "UCB-GLM":    ["green", "solid", "OFUGLB-e"],
        "GLM-TS":     ["black", "solid", "LogTS"],
        "LogPHE":     ["purple", "solid", "LogPHE"],
        "LogFPL":     ["purple", "solid", "LogFPL"],
        "RandUCBLog": ["red", "solid", "RandUCBLog"],
        "GLM-FP":     ["blue", "solid", "GLM-FP (ours)"],
        "RS-GLinCB":  ["blue", "dashed", "RS-GLinCB"]
    }

    alg_styles = [(alg, *alg_map[alg]) for alg in algorithms]

    # environments => list of (env_name, d, T)
    norm_envs = []
    for e in environments:
        env_name, d, T = e
        norm_envs.append((env_name, d, T))
    environments = norm_envs

    n_env = len(environments)
    fig, axs = plt.subplots(1, n_env, figsize=(5*n_env+2, 3), squeeze=False)
    handles, labels = [], []

    for idx, (env_name, d, T) in enumerate(environments):
        step = np.arange(1, T+1)
        ax = axs[0, idx]
        res_dir = os.path.join(".", "Results", filename, env_name)

        for alg_name, color, style, label in alg_styles:
            data = np.loadtxt(os.path.join(res_dir, alg_name + ".csv"), delimiter=",")
            if data.ndim == 1: data = data.reshape(-1,1)
            mean_cr = data.mean(axis=1)
            std_cr = data.std(axis=1, ddof=1)/np.sqrt(data.shape[1])

            ax.plot(step, mean_cr, color=color, linestyle=linestyle2dashes(style), label=label)
            pts = np.linspace(0, T-1, 10, dtype=int)
            ax.errorbar(step[pts], mean_cr[pts], yerr=std_cr[pts], fmt='o',
                        color=color, capsize=3, capthick=1, markersize=2)

        ax.set_title(f"$d={d}, T={T}$" if d else env_name)
        ax.set_xlabel("Round $t$")
        if idx == 0: ax.set_ylabel("Cumulative Regret")
        ax.set_xlim(1, T)
        ax.grid(True)
        ylim = mean_cr[-1] * 1.5
        # ax.set_ylim(0, ylim)

        h, l = ax.get_legend_handles_labels()
        for hi, li in zip(h, l):
            if li not in labels:
                handles.append(hi)
                labels.append(li)

    fig.legend(handles, labels, loc='upper center', ncol=len(labels),
               frameon=True, bbox_to_anchor=(0.5, 1.15), fontsize='small',)

    plot_dir = os.path.join(".", "Plots")
    os.makedirs(plot_dir, exist_ok=True)
    os.makedirs(os.path.join(plot_dir, "linear"), exist_ok=True)
    os.makedirs(os.path.join(plot_dir, "logistic"), exist_ok=True)

    fname = os.path.join(plot_dir, f"{filename}.pdf")
    plt.savefig(fname, dpi=300, bbox_inches='tight')
    plt.show()
