/**
File:		NECore/DataTypes/NeArray4.h

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

#pragma once

#include <NeCoreLib.h>
#include <cstddef>
#include <vector>

namespace NeuralEngine
{

	template <typename T>

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	/// 	The Array4 class represents a 4-dimensional array that minimizes the number
	/// 	of new and delete calls.  The T objects are stored in a contiguous array.		
	/// </summary>
	///
	/// <remarks>	Hmetal T, 04.08.2016. </remarks>
	////////////////////////////////////////////////////////////////////////////////////////////////////
	class Array4
	{
	public:

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructor. </summary>
		/// 
		/// <note>
		///		The first constructor generates an array of objects that
		///		are owned by Array4.  The array has bound0 columns,
		///		bound1 rows, bound2 slices, and bound3 cuboids.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="bound0">	The bound 0. </param>
		/// <param name="bound1">	The first bound. </param>
		/// <param name="bound2">	The second bound. </param>
		/// <param name="bound3">	The third bound. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		Array4(size_t bound0, size_t bound1, size_t bound2, size_t bound3);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructor. </summary>
		/// 
		/// <note>
		///		The second constructor is given an array of
		///		objects that are owned by the caller.  The array has bound0 columns,
		///		bound1 rows, bound2 slices, and bound3 cuboids.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="bound0"> 	The bound 0. </param>
		/// <param name="bound1"> 	The first bound. </param>
		/// <param name="bound2"> 	The second bound. </param>
		/// <param name="bound3"> 	The third bound. </param>
		/// <param name="objects">	[in,out] If non-null, the objects. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		Array4(size_t bound0, size_t bound1, size_t bound2, size_t bound3, T* objects);


		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Default constructor. </summary>
		/// 
		/// <note>
		///		Support for dynamic resizing, copying, or moving.  If 'other' does
		/// 	not own the original 'objects', they are not copied by the assignment
		/// 	operator.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		Array4();
		Array4(Array4 const&);
		Array4& operator=(Array4 const&);
		Array4(Array4&&);
		Array4& operator=(Array4&&);

		// Access to the array.  Sample usage is
		//   Array4<T> myArray(5, 4, 3, 2);
		//   T*** cuboid1 = myArray[1];
		//   T** cuboid1Slice2 = myArray[1][2];
		//   T* cuboid1Slice2Row3 = myArray[1][2][3];
		//   T cuboid1Slice2Row3Col4 = myArray[1][2][3][4];
		inline size_t GetBound0() const;
		inline size_t GetBound1() const;
		inline size_t GetBound2() const;
		inline size_t GetBound3() const;
		inline T** const* operator[] (int cuboid) const;
		inline T*** operator[] (int cuboid);

	private:
		void SetPointers(T* objects);
		void SetPointers(Array4 const& other);

		size_t mBound0, mBound1, mBound2, mBound3;
		std::vector<T> mObjects;
		std::vector<T*> mIndirect1;
		std::vector<T**> mIndirect2;
		std::vector<T***> mIndirect3;
	};
#include "NeArray4.inl"
}
