/**
File:		NECore/DataTypes/NeMinHeap.h

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

#pragma once

#include <NeCoreLib.h>
#include <vector>

namespace NeuralEngine
{

	template <typename KeyType, typename ValueType>

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// <summary>	Minimum heap binary tree. </summary>
	/// 
	/// <note>
	///		A min-heap is a binary tree whose nodes have weights and with the
	/// 	constraint that the weight of a parent node is less than or equal to the
	/// 	weights of its children.  This data structure may be used as a priority
	/// 	queue.  If the std::priority_queue interface suffices for your needs, use
	/// 	that instead.  However, for some geometric algorithms, that interface is
	/// 	insufficient for optimal performance.  For example, if you have a polyline
	/// 	vertices that you want to decimate, each vertex's weight depends on its
	/// 	neighbors' locations.  If the minimum-weight vertex is removed from the
	/// 	min-heap, the neighboring vertex weights must be updated--something that
	/// 	is O(1) time when you store the vertices as a doubly linked list.  The
	/// 	neighbors are already in the min-heap, so modifying their weights without
	/// 	removing then from--and then reinserting into--the min-heap requires they
	/// 	must be moved to their proper places to restore the invariant of the
	/// 	min-heap.  With std::priority_queue, you have no direct access to the
	/// 	modified vertices, forcing you to search for those vertices, remove them,
	/// 	update their weights, and re-insert them.  The min-heap implementation here
	/// 	does support the update without removal and reinsertion.
	/// 
	/// 	The ValueType represents the weight and it must support comparisons
	/// 	"<" and "<=".  Additional information can be stored in the min-heap for
	/// 	convenient access; this is stored as the KeyType.  In the (open) polyline
	/// 	decimation example, the KeyType is a structure that stores indices to
	/// 	a vertex and its neighbors.  The following code illustrates the creation
	/// 	and use of the min-heap.  The Weight() function is whatever you choose to
	/// 	guide which vertices are removed first from the polyline.
	/// 
	/// 	   struct Vertex { int previous, current, next; };
	/// 	   int numVertices = <number of polyline vertices>;
	/// 	   std::vector<Vector<N, Real>> positions(numVertices);
	/// 	   <assign all positions[*]>;
	/// 	   MinHeap<Vertex, Real> minHeap(numVertices);
	/// 	   std::vector<MinHeap<Vertex, Real>::Record*> records(numVertices);
	/// 	   for (int i = 0; i < numVertices; ++i)
	/// 	   {
	/// 	       Vertex vertex;
	/// 	       vertex.previous = (i + numVertices - 1) % numVertices;
	/// 	       vertex.current = i;
	/// 	       vertex.next = (i + 1) % numVertices;
	/// 	       records[i] = minHeap.Insert(vertex, Weight(positions, vertex));
	/// 	   }
	/// 
	/// 	   while (minHeap.GetNumElements() >= 2)
	/// 	   {
	/// 	       Vertex vertex;
	/// 	       Real weight;
	/// 	       minHeap.Remove(vertex, weight);
	/// 	       <consume the 'vertex' according to your application's needs>;
	/// 
	/// 	       // Remove 'vertex' from the doubly linked list.
	/// 	       Vertex& vp = records[vertex.previous]->key;
	/// 	       Vertex& vc = records[vertex.current]->key;
	/// 	       Vertex& vn = records[vertex.next]->key;
	/// 	       vp.next = vc.next;
	/// 	       vn.previous = vc.previous;
	/// 
	/// 	       // Update the neighbors' weights in the min-heap.
	/// 	       minHeap.Update(records[vertex.previous], Weight(positions, vp));
	/// 	       minHeap.Update(records[vertex.next], Weight(positions, vn));
	/// 	   }
	/// </note>
	///
	/// <remarks>	Hmetal T, 04.08.2016. </remarks>
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class MinHeap
	{
	public:
		struct Record
		{
			KeyType key;
			ValueType value;
			int index;
		}; 

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Copy constructor. </summary>
		/// 
		/// <note>
		///		Construction.  The record 'value' members are uninitialized for native
		/// 	types chosen for ValueType.  If ValueType is of class type, then the
		/// 	default constructor is used to set the 'value' members.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="minHeap">	The minimum heap. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MinHeap(MinHeap const& minHeap);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Constructor. </summary>
		/// 
		/// <note>
		///		Construction.  The record 'value' members are uninitialized for native
		/// 	types chosen for ValueType.  If ValueType is of class type, then the
		/// 	default constructor is used to set the 'value' members.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="maxElements">	(Optional) the maximum elements. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		MinHeap(int maxElements = 0);

		// Assignment.
		MinHeap& operator=(MinHeap const& minHeap);
 

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Resets the given maxElements. </summary>
		/// 
		/// <note>
		///		Clear the min-heap so that it has the specified max elements,
		///		mNumElements is zero, and mPointers are set to the natural ordering
		/// 	of mRecords.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="maxElements">	The maximum elements. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void Reset(int maxElements);

		// 
		// 

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>
		/// 	Get the remaining number of elements in the min-heap.  This number is
		/// 	in the range {0..maxElements}.				
		/// </summary>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <returns>	The number elements. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		inline int GetNumElements() const;

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>
		/// 	Get the root of the min-heap.  The return value is 'true' whenever the
		/// 	min-heap is not empty.  This function reads the root but does not		
		/// 	remove the element from the min-heap.			
		/// </summary>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="key">  	[in,out] The key. </param>
		/// <param name="value">	[in,out] The value. </param>
		///
		/// <returns>	true if it succeeds, false if it fails. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		inline bool GetMinimum(KeyType& key, ValueType& value) const;


		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Insert into the min-heap the 'value' that corresponds to the 'key'. </summary>
		/// 
		/// <note>
		///		The return value is a pointer to the heap record that stores a copy of
		/// 	'value', and the pointer value is constant for the life of the min-heap.
		/// 	If you must update a member of the min-heap, say, as illustrated in the
		/// 	polyline decimation example, pass the pointer to Update:
		/// 	   auto* valueRecord = minHeap.Insert(key, value);
		/// 	   <do whatever>;
		/// 	   minHeap.Update(valueRecord, newValue).
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="key">  	The key. </param>
		/// <param name="value">	The value. </param>
		///
		/// <returns>	null if it fails, else a pointer to a Record. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		Record* Insert(KeyType const& key, ValueType const& value);

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Remove the root of the heap and return its 'key' and 'value members. </summary>
		/// 
		/// <note>
		///		The root contains the minimum value of all heap elements.  The return
		/// 	value is 'true' whenever the min-heap was not empty before the Remove
		/// 	call.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="key">  	[in,out] The key. </param>
		/// <param name="value">	[in,out] The value. </param>
		///
		/// <returns>	true if it succeeds, false if it fails. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		bool Remove(KeyType& key, ValueType& value);


		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	The value of a heap record must be modified through this function call. </summary>
		/// 
		/// <note>
		///		The side effect is that the heap is updated accordingly to restore the
		/// 	data structure to a min-heap.  The input 'record' should be a pointer
		/// 	returned by Insert(value); see the comments for the Insert() function.
		/// </note>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <param name="record">	[in,out] If non-null, the record. </param>
		/// <param name="value"> 	The value. </param>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		void Update(Record* record, ValueType const& value);

		// 
		// 

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	
		///		Support for debugging.  The functions test whether the data structure
		/// 	is a valid min-heap.
		/// </summary>
		///
		/// <remarks>	Hmetal T, 04.08.2016. </remarks>
		///
		/// <returns>	true if valid, false if not. </returns>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		bool IsValid() const;

	private:
		// A 2-level storage system is used.  The pointers have two roles.
		// Firstly, they are unique to each inserted value in order to support
		// the Update() capability of the min-heap.  Secondly, they avoid
		// potentially expensive copying of Record objects as sorting occurs in
		// the heap.
		int mNumElements;
		std::vector<Record> mRecords;
		std::vector<Record*> mPointers;
	};
#include "NeMinHeap.inl"
}
