/**
File:		NECore/Time/NeTimer.cpp

Author:		
Email:		
Site:       

Copyright (c) 2016 . All rights reserved.
*/

#include <NeCorePCH.h>
#include <Core/NeTimer.h>

namespace NeuralEngine
{

#if defined(NE_NEEDS_64_BIT_CLOCK)

#include <cmath>
#include <windows.h>

	Timer::Timer()
		:
		mFrequency(0),
		mInitialTicks(0),
		mInvFrequency(0.0)
	{
		LARGE_INTEGER frequency = { 1 }, counter = { 0 };
		QueryPerformanceFrequency(&frequency);
		QueryPerformanceCounter(&counter);
		mFrequency = static_cast<int64_t>(frequency.QuadPart);
		mInitialTicks = static_cast<int64_t>(counter.QuadPart);
		mInvFrequency = 1.0 / static_cast<double>(mFrequency);
	}

	int64_t Timer::GetNanoseconds() const
	{
		int64_t ticks = GetTicks();
		double seconds = mInvFrequency * static_cast<double>(ticks);
		int64_t nanoseconds = static_cast<int64_t>(ceil(1000000000.0 * seconds));
		return nanoseconds;
	}

	int64_t Timer::GetMicroseconds() const
	{
		int64_t ticks = GetTicks();
		double seconds = mInvFrequency * static_cast<double>(ticks);
		int64_t microseconds = static_cast<int64_t>(ceil(1000000.0 * seconds));
		return microseconds;
	}

	int64_t Timer::GetMilliseconds() const
	{
		int64_t ticks = GetTicks();
		double seconds = mInvFrequency * static_cast<double>(ticks);
		int64_t milliseconds = static_cast<int64_t>(ceil(1000.0 * seconds));
		return milliseconds;
	}

	double Timer::GetSeconds() const
	{
		int64_t ticks = GetTicks();
		double seconds = mInvFrequency * static_cast<double>(ticks);
		return seconds;
	}

	void Timer::Reset()
	{
		LARGE_INTEGER counter = { 0 };
		QueryPerformanceCounter(&counter);
		mInitialTicks = static_cast<int64_t>(counter.QuadPart);
	}

	int64_t Timer::GetTicks() const
	{
		LARGE_INTEGER counter = { 0 };
		QueryPerformanceCounter(&counter);
		return static_cast<int64_t>(counter.QuadPart) - mInitialTicks;
	}

#else

	Timer::Timer()
	{
		Reset();
	}

	int64_t Timer::GetNanoseconds() const
	{
		auto currentTime = std::chrono::high_resolution_clock::now();
		return std::chrono::duration_cast<std::chrono::nanoseconds>(
			currentTime - mInitialTime).count();
	}

	int64_t Timer::GetMicroseconds() const
	{
		auto currentTime = std::chrono::high_resolution_clock::now();
		return std::chrono::duration_cast<std::chrono::microseconds>(
			currentTime - mInitialTime).count();
	}

	int64_t Timer::GetMilliseconds() const
	{
		auto currentTime = std::chrono::high_resolution_clock::now();
		return std::chrono::duration_cast<std::chrono::milliseconds>(
			currentTime - mInitialTime).count();
	}

	double Timer::GetSeconds() const
	{
		int64_t msecs = GetMilliseconds();
		return static_cast<double>(msecs) / 1000.0;
	}

	void Timer::Reset()
	{
		mInitialTime = std::chrono::high_resolution_clock::now();
	}

#endif
}