/**
File:		MachineLearning/Util/CommonUtil.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <Core/NeLogger.h>
#include <MachineLearning/FgArrayFireSerialization.h>



namespace NeuralEngine
{
	namespace MachineLearning
	{
		template<class T>
		struct is_Scalar
			: std::integral_constant<
			bool,
			std::is_same<double, typename std::remove_cv<T>::type>::value> {};

		template<class T>
		struct is_float
			: std::integral_constant<
			bool,
			std::is_same<float, typename std::remove_cv<T>::type>::value> {};

		template<typename Scalar>
		class NE_IMPEXP CommonUtil
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the Euclidean norm for a vector. </summary>
			///
			/// <remarks>	 Admin, 3/27/2017. </remarks>
			///
			/// <param name="a">	[in,out] The af::array to process. </param>
			///
			/// <returns>	A Scalar. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static Scalar Euclidean(const af::array& a);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the Euclidean norm for two vectors. </summary>
			///
			/// <remarks>	Hmetal T, 29/03/2018. </remarks>
			///
			/// <param name="a">	[in,out] The af::array to process. </param>
			/// <param name="b">	[in,out] The af::array to process. </param>
			///
			/// <returns>	A Scalar. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array Euclidean(const af::array& a, const af::array& b);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the Square Euclidean norm for two vectors. </summary>
			///
			/// <remarks>	 Admin, 3/28/2017. </remarks>
			///
			/// <param name="a">	[in,out] The af::array to process. </param>
			///
			/// <returns>	A Scalar. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static Scalar SquareEuclidean(const af::array& a);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the Square Euclidean norm for a vector. </summary>
			///
			/// <remarks>	Hmetal T, 29/03/2018. </remarks>
			///
			/// <param name="a">	[in,out] The af::array to process. </param>
			/// <param name="b">	[in,out] The af::array to process. </param>
			///
			/// <returns>	A Scalar. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array SquareEuclidean(const af::array& a, const af::array& b);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the maximum value among three values. </summary>
			///
			/// <remarks>	Hmetal T, 03.04.2017. </remarks>
			///
			/// <param name="a">	The first value <c>a</c>. </param>
			/// <param name="b">	The second value <c>b</c>. </param>
			/// <param name="c">	The third value <c>c</c>. </param>
			///
			/// <returns>
			/// 	The maximum value among <paramref name="a"/>,
			/// 	<paramref name="b"/> and <paramref name="c"/>.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static Scalar Max(Scalar a, Scalar b, Scalar c);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Covariance. </summary>
			///
			/// <remarks>	Hmetal T, 14.04.2017. </remarks>
			///
			/// <param name="a">	[in,out] Data matrix with row observation vectors. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array Covariance(const af::array& M);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Correlation coefficients. </summary>
			///
			/// <remarks>
			/// 	calculates a matrix R of correlation coefficients for an array X, in which each row is an
			/// 	observation and each column is a variable.
			/// </remarks>
			///
			/// <param name="inX">	[in,out] The in x coordinate. </param>
			/// <param name="inY">	[in,out] (Optional) the in y coordinate. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array CorrelationCoefficients(const af::array& inX, const af::array& inY = af::array());

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets standardized Square Euclidean distance. </summary>
			///
			/// <remarks>
			/// 			Computation has the form of,
			/// 			
			/// 					\|\mathbf{x1}-\mathbf{x2}\|^2.
			/// 			
			///				 Admin, 5/24/2017. 	
			/// </remarks>
			///
			/// <param name="inX1">	[in,out] The first x coordinate. </param>
			/// <param name="inX2">	[in,out] The second x coordinate. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array SquareDistance(const af::array& inX1, const af::array& inX2);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Unscaled distance. </summary>
			///
			/// <remarks>
			/// 			Computation has the form of,
			/// 			
			/// 					\sqrt{\|\mathbf{x1}-\mathbf{x2}\|^2}.
			/// 			
			///				 Admin, 5/24/2017. 	
			/// </remarks>
			///
			/// <param name="inX1">	[in,out] The first x coordinate. </param>
			/// <param name="inX2">	[in,out] The second x coordinate. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array UnscaledDistance(const af::array& inX1, const af::array& inX2);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Scaled distance. </summary>
			///
			/// <remarks>
			/// 			Computation has the form of,
			/// 			
			/// 					\sqrt{\sum_{d=1}^D{(x1_d-x2_d)^2/v_d}}.
			/// 			
			///				 Admin, 5/24/2017. 	
			/// </remarks>
			///
			/// <param name="inX1">		   	The first x coordinate. </param>
			/// <param name="inX2">		   	The second x coordinate. </param>
			/// <param name="inLengtScale">	The lengt scale. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array ScaledDistance(const af::array& inX1, const af::array& inX2, const af::array& inLengtScale);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Normal probability density function. </summary>
			///
			/// <remarks>
			/// 	Normal PDF	is defined as follows:
			/// 	
			/// 		$$f(x;mu,sigma)=\frac{1}{x\sqrt{2\pi\sigma^2}}\exp{-\frac{(x-mu)^2}{2\sigma^2}}.$$	
			/// 			
			/// 	Hmetal T, 22/11/2019. 
			/// </remarks>
			///
			/// <param name="inX">	  	The in x coordinate. </param>
			/// <param name="inMu">   	(Optional) The in mu. </param>
			/// <param name="inSigma">	(Optional) The in sigma. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array NormalPDF(const af::array& inX, const af::array& inMu = af::array(), const af::array& inSigma = af::array());

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Log normal probability density function. </summary>
			///
			/// <remarks>
			/// 	Log normal PDF	is defined as follows:
			/// 	
			/// 		$$f(x;mu,sigma)=\frac{1}{x\sqrt{2\pi\sigma^2}}\exp{-\frac{(\log(x)-mu)^2}{2\sigma^2}}.$$	
			/// 			
			/// 	Hmetal T, 22/11/2019. 
			/// </remarks>
			///
			/// <param name="inX">	  	The in x coordinate. </param>
			/// <param name="inMu">   	(Optional) The in mu. </param>
			/// <param name="inSigma">	(Optional) The in sigma. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array LogNormalPDF(const af::array& inX, const af::array& inMu = af::array(), const af::array& inSigma = af::array());

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Normal cumulative distribution function.. </summary>
			///
			/// <remarks>
			/// 	Normal CDF	is defined as follows:
			/// 	
			/// 		$$f(x;mu,sigma)=\frac{1}{2}\left[1 + erf\left(\frac{x-mu}{\sigma\sqrt{2}}\right)\right].$$
			/// 			
			/// 	Hmetal T, 22/11/2019. 
			/// </remarks>
			///
			/// <param name="inX">	  	The in x coordinate. </param>
			/// <param name="inMu">   	(Optional) The in mu. </param>
			/// <param name="inSigma">	(Optional) The in sigma. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array NormalCDF(const af::array& inX, const af::array& inMu = af::array(), const af::array& inSigma = af::array());

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Log normal cumulative distribution function. </summary>
			///
			/// <remarks>	
			/// 	Log normal CDF	is defined as follows:
			/// 	
			/// 		$$f(x;mu,sigma)=\frac{1}{2}\left[1 + erf\left(\frac{\ln{x}-mu}{\sigma\sqrt{2}}\right)\right].$$
			/// 			
			/// 	Hmetal T, 22/11/2019. 
			/// </remarks>
			///
			/// <param name="inX">	  	The in x coordinate. </param>
			/// <param name="inMu">   	(Optional) The in mu. </param>
			/// <param name="inSigma">	(Optional) The in sigma. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array LogNormalCDF(const af::array& inX, const af::array& inMu = af::array(), const af::array& inSigma = af::array());

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes indexes of upper triangular matrix. </summary>
			///
			/// <remarks>	Hmetal T, 03/07/2018. </remarks>
			///
			/// <param name="size">	Size of the matrix. </param>
			///
			/// <returns>	An af::array of indexes. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array TriUpperIdx(int size, int dimension = 0);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes indexes of lower triangular matrix. </summary>
			///
			/// <remarks>	Hmetal T, 03/07/2018. </remarks>
			///
			/// <param name="size">	Size of the matrix. </param>
			///
			/// <returns>	An af::array of indexes. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array TriLowerIdx(int size);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Indexes of matrix diagonal. </summary>
			///
			/// <remarks>	Hmetal T, 23/04/2019. </remarks>
			///
			/// <param name="size">	Size of the matrix. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array DiagIdx(int size);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Implementation of NumPy's Linspace. </summary>
			///
			/// <remarks>	Hmetal T, 05/07/2018. </remarks>
			///
			/// <param name="start">   	The start value. </param>
			/// <param name="stop">	   	The stop vale. </param>
			/// <param name="num">	   	Number of steps. </param>
			/// <param name="endpoint">	(Optional) true to endpoint. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array LinSpace(Scalar start, Scalar stop, int num, bool endpoint = true);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Cholesky decomposition. </summary>
			///
			/// <remarks>	
			/// 	If needed, adds jitter to the diagonal to ensure semipositive definitness.
			/// 	
			/// 	Hmetal T, 05/07/2018. 
			/// </remarks>
			///
			/// <param name="inA">	[in,out] The in a. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array JitChol(const af::array& inA);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Log determinant of the input matrix. </summary>
			///
			/// <remarks>	Hmetal T, 05/07/2018. </remarks>
			///
			/// <param name="inA">	The in a. </param>
			///
			/// <returns>	A Scalar. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static Scalar LogDet(af::array inA);

			static void MergeMaps(std::map<std::string, af::array>& lhs, const std::map<std::string, af::array>& rhs);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Sort rows. </summary>
			///
			/// <remarks>	Hmetal T, 03.05.2019. </remarks>
			///
			/// <param name="inA">	The in a. </param>
			///
			/// <returns>	The sorted rows. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array SortRows(af::array inA);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Joins matrices and vectors along specific dimension. </summary>
			///
			/// <remarks>	Hmetal T, 06.05.2019. </remarks>
			///
			/// <param name="inA">	The in a. </param>
			/// <param name="inB">	The in b. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array Join(const af::array& inA, const af::array& inB, int dimension = 0);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Linear solve usingQR decomposition. </summary>
			///
			/// <remarks>	
			/// 	Equivalent of Matlabs mldivide x = A\b.
			/// 	
			/// 	Hmetal T, 27/05/2019. 
			/// </remarks>
			///
			/// <param name="A">	[in,out] The af::array to process. </param>
			/// <param name="b">	[in,out] The af::array to process. </param>
			///
			/// <returns>	Solution x. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array SolveQR(const af::array& A, const af::array& b);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes the pseudo inverse of a matrix. </summary>
			///
			/// <remarks>	Hmetal T, 06/01/2020. </remarks>
			///
			/// <param name="inA">	[in,out] The in symetric matrix. </param>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array PDInverse(const af::array& inA);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Loads an array from a text file. </summary>
			///
			/// <remarks>	Hmetal T, 20/12/2019. </remarks>
			///
			/// <param name="filename">	Filename of the file. </param>
			///
			/// <returns>	The text. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::array ReadTXT(std::string filename, char delimiter = ' ');

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Writes an array to a text file. </summary>
			///
			/// <remarks>	Hmetal T, 20/12/2019. </remarks>
			///
			/// <param name="data">			The data array to write. </param>
			/// <param name="filename"> 	Filename of the file. </param>
			/// <param name="delimiter">	(Optional) The delimiter. </param>
			///
			/// <returns>	True if it succeeds, false if it fails. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static bool WriteTXT(const af::array& data, std::string filename, char delimiter = ' ');

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Query if 'a' is equal to 'b'. </summary>
			///
			/// <remarks>	Hmetal T, 29/04/2021. </remarks>
			///
			/// <param name="a">	An af::array to process. </param>
			/// <param name="b">	An af::array to process. </param>
			///
			/// <returns>	True if equal, false if not. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static bool IsEqual(const af::array& a, const af::array& b);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Checks af::array precision flag. </summary>
			///
			/// <remarks>	Hmetal T, 17/06/2019. </remarks>
			///
			/// <returns>	A dtype. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			static af::dtype CheckDType();
		};
	}
}