﻿/**
File:		MachineLearning/Util/FgEigenvalueDecomposition.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#undef min
#undef max

#include <NeMachineLearningLib.h>
#include <arrayfire.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Eigenvalue Decomposition. </summary>
		/// 
		/// <note>
		///		In linear algebra, eigendecomposition or sometimes spectral decomposition is the 
		///		factorization of a matrix into a canonical form, whereby the matrix is represented 
		///		in terms of its eigenvalues and eigenvectors. Only diagonalizable matrices can be 
		///		factorized in this way.
		/// </note>
		///
		/// <remarks>	Hmetal T, 08.03.2017. </remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		class NE_IMPEXP EigenvalueDecomposition
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Constructor. </summary>
			///
			/// <remarks>	Hmetal T, 14.04.2017. </remarks>
			///
			/// <param name="A">	[in,out] Positive definit matrix. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			EigenvalueDecomposition(af::array& A);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	Hmetal T, 14.04.2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			~EigenvalueDecomposition();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Computes the eigenvalues and -vectors in decending order</summary>
			/// 
			/// <note>
			///		Let A be a square (N×N) matrix with N linearly independent eigenvectors, 
			///		q_i ( i = 1 , … , N ). Then A can be factorized as
			///
			///			A = Q Λ Q^(− 1) 
			///
			///		where Q is the square(N×N) matrix whose ith column is the eigenvector q_i of A and Λ 
			///		is the diagonal matrix whose diagonal elements are the corresponding eigenvalues, 
			///		i.e., Λ_(i i) = λ_i. Note that only diagonalizable matrices can be factorized in this way.
			/// </note>
			///
			/// <remarks>	Hmetal T, 08.03.2017. </remarks>
			///
			/// <param name="A">			[in,out] Positiv semidefinit input matrix. </param>
			/// <param name="outEigvec">	[in,out] Corresponding eigenvectors. </param>
			///
			/// <returns>	An af::array of eigenvalues. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void Compute(af::array& A);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the eigenvalues in decending order. </summary>
			///
			/// <remarks>	Hmetal T, 14.04.2017. </remarks>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			af::array Eigenvalues();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the corresponding eigenvectors. </summary>
			///
			/// <remarks>	Hmetal T, 14.04.2017. </remarks>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			af::array Eigenvectors();

		private:
			void compute();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Nonsymmetric reduction to Hessenberg form. </summary>
			///
			/// <remarks>	
			/// 	This is derived from the Algol procedures orthes and ortran,
			///		by Martin and Wilkinson, Handbook for Auto. Comp.,
			///		Vol.ii-Linear Algebra, and the corresponding
			///		Fortran subroutines in EISPACK. 
			///	</remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void orthes();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Nonsymmetric reduction from Hessenberg to real Schur form. </summary>
			///
			/// <remarks>	
			/// 	This is derived from the Algol procedure hqr2,
			///		by Martin and Wilkinson, Handbook for Auto. Comp.,
			///		Vol.ii-Linear Algebra, and the corresponding
			///		Fortran subroutine in EISPACK. 
			///	</remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			void hqr2();

			void cdiv(double xr, double xi, double yr, double yi);

			int n;
			double cdivr, cdivi;

			double *d, *e, *ort;
			double **V, **H;

			af::array fV, fd;

			template<typename _Tp>
			_Tp *alloc_1d(int m)
			{
				return new _Tp[m];
			}

			template<typename _Tp>
			_Tp *alloc_1d(int m, _Tp val)
			{
				_Tp *arr = alloc_1d<_Tp>(m);
				for (int i = 0; i < m; i++)
					arr[i] = val;
				return arr;
			}

			template<typename _Tp>
			_Tp **alloc_2d(int m, int n)
			{
				_Tp **arr = new _Tp*[m];
				for (int i = 0; i < m; i++)
					arr[i] = new _Tp[n];
				return arr;
			}

			template<typename _Tp>
			_Tp **alloc_2d(int m, int n, _Tp val)
			{
				_Tp **arr = alloc_2d<_Tp>(m, n);
				for (int i = 0; i < m; i++) {
					for (int j = 0; j < n; j++) {
						arr[i][j] = val;
					}
				}
				return arr;
			}
		};
	}
}