/**
File:		MachineLearning/Models/GPModels/FgDeepGPBaseModel.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgGPNode.h>
#include <MachineLearning/FgGPBaseModel.h>
#include <MachineLearning/FgBackconstraints.h>
#include <boost/serialization/map.hpp>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Style variable. </summary>
			///
			/// <remarks>	Hmetal T, 25/09/2020. </remarks>
			///
			/// <typeparam name="Scalar">	Type of the scalar. </typeparam>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP Style
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <param name="name">	The name of the style. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				Style(std::string name);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	Hmetal T, 21/03/2021. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				Style() { };

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~Style();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Adds a sub-style. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <param name="name">			The name of the sub-style. </param>
				/// <param name="numFrames">	Number of frames. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void AddSubstyle(std::string name, int numFrames);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of sub-styles. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The number substyles. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumSubstyles();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of inducing sub-styles. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The number substyles. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumInducingSubstyles();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the style variable. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The style. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetStyle();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the sub-styles. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	Null if it fails, else the sub-style names and associated indexes. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				std::map<std::string, int> GetSubstyles();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets style variable expanded to match associated /f$\matchbf{x}$/f. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The style expanded. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetStyleExpanded();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets inducing style expanded. </summary>
				///
				/// <remarks>	Hmetal T, 09/11/2020. </remarks>
				///
				/// <returns>	The inducing style expanded. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetInducingStyleExpanded();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets style index. </summary>
				///
				/// <remarks>	Hmetal T, 28/09/2020. </remarks>
				///
				/// <returns>	The style index. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				const af::array GetStyleIndex();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets inducing style index. </summary>
				///
				/// <remarks>	Hmetal T, 06/11/2020. </remarks>
				///
				/// <param name="indx">	The indx. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetInducingStyleIndex(const af::array& indx);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets factor x coordinate 1. </summary>
				///
				/// <remarks>	Hmetal T, 28/09/2020. </remarks>
				///
				/// <returns>	The factor x coordinate 1. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void GetFactorsExpanded(af::array& X1, af::array& X2);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the name of the style. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The name. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				std::string GetName();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number parameters. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				int GetNumParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of inducing parameters. </summary>
				///
				/// <remarks>	Hmetal T, 06/11/2020. </remarks>
				///
				/// <returns>	The number inducing parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				int GetNumInducingParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	The parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <param name="param">	[in,out] The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetParameters(const af::array& param);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets inducing parameters. </summary>
				///
				/// <remarks>	Hmetal T, 09/11/2020. </remarks>
				///
				/// <returns>	The inducing parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetInducingParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets inducing parameters. </summary>
				///
				/// <remarks>	Hmetal T, 09/11/2020. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetInducingParameters(const af::array& param);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 28/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void UpdatePosterior(const af::array& postX1, const af::array& postX2);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets gradient collapsed. </summary>
				///
				/// <remarks>	Hmetal T, 28/09/2020. </remarks>
				///
				/// <param name="factorFradient">	[in,out] The factor fradient. </param>
				///
				/// <returns>	The gradient collapsed. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetGradientCollapsed(const af::array& factorGradient);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets style inducing gradient collapsed. </summary>
				///
				/// <remarks>	HmetalT, 06/11/2020. </remarks>
				///
				/// <param name="styleGradient">	The style gradient. </param>
				///
				/// <returns>	The inducing gradient collapsed. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetInducingGradientCollapsed(const af::array& styleGradient);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes the factors. </summary>
				///
				/// <remarks>	Hmetal T, 28/09/2020. </remarks>
				///
				/// <param name="expandedX1">	[in,out] The first expanded x coordinate. </param>
				/// <param name="expandedX1">	[in,out] The first expanded x coordinate. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void InitFactors(const af::array& expandedX1, const af::array& expandedX2);

			protected:
				af::array afStyleIdx;
				af::array afStyleInducingIdx;
				af::array afStyle;
				af::array afInducingStyle;
				af::array afFactorX1;
				af::array afFactorX2;
				af::array afPosteriorX1;
				af::array afPosteriorX2;
				std::map<std::string, int> mSubStyles;

				std::string sName;

				af::dtype m_dType;			//!< floating point precision flag for af::array

			private:
				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					//ar& boost::serialization::base_object<GPLVMBaseModel<Scalar>>(*this);

					ar& BOOST_SERIALIZATION_NVP(afStyleIdx);
					ar& BOOST_SERIALIZATION_NVP(afStyleInducingIdx);
					ar& BOOST_SERIALIZATION_NVP(afStyle);
					ar& BOOST_SERIALIZATION_NVP(afInducingStyle);
					ar& BOOST_SERIALIZATION_NVP(mSubStyles);
					ar& BOOST_SERIALIZATION_NVP(sName);
					ar& BOOST_SERIALIZATION_NVP(m_dType);
				}
			};

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	
			/// 	Base class with abstract and basic function definitions. All deep GP models will be derived 
			/// 	from this class.			
			/// </summary>
			///
			/// <remarks>	HmetalT, 26.10.2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP GPLVMBaseModel : public GPBaseModel<Scalar>, public GPNode<Scalar>
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Constructor. </summary>
				///
				/// <remarks>	, 26.03.2018. </remarks>
				///
				/// <param name="Y">			  	The data af::array to process. </param>
				/// <param name="latentDimension">	The training inputs. </param>
				/// <param name="priorMean">	  	(Optional) The description for one hidden layer. </param>
				/// <param name="priorVariance">  	(Optional) The prior variance. </param>
				/// <param name="numInducing">	  	(Optional) Number of inducings. </param>
				/// <param name="lType">		  	(Optional) the loglik type. </param>
				/// <param name="emethod">		  	(Optional) The emethod. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				GPLVMBaseModel(const af::array& Y, int latentDimension, Scalar priorMean = 0.0, Scalar priorVariance = 1.0,
					LogLikType lType = LogLikType::Gaussian, XInit emethod = XInit::pca);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	Hmetal T, 31/03/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				GPLVMBaseModel();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	, 15.05.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~GPLVMBaseModel();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Optimizes the model parameters for best fit. </summary>
				///
				/// <remarks>	Hmetal T, 29.11.2017. </remarks>
				///
				/// <param name="method">			(Optional) the optimization method. </param>
				/// <param name="tol">				(Optional) the tolerance. </param>
				/// <param name="reinit_hypers">	(Optional) true to re hypers. </param>
				/// <param name="maxiter">			(Optional) max iterations. </param>
				/// <param name="mb_size">			(Optional) batch size. </param>
				/// <param name="LSType">			(Optional) linesearch type. </param>
				/// <param name="disp">				(Optional) true to disp. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void Optimise(
					OptimizerType method = L_BFGS,
					Scalar tol = 0.0,
					bool reinit_hypers = true,
					int maxiter = 1000,
					int mb_size = 0,
					LineSearchType lsType = MoreThuente,
					bool disp = true,
					int* cycle = nullptr
				) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes the model. </summary>
				///
				/// <remarks>	Hmetal T, 29.11.2017. </remarks>
				///
				/// <returns>	true if it succeeds, false if it fails. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual bool Init(af::array& mx);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes the model. </summary>
				///
				/// <remarks>	Hmetal T, 29.11.2017. </remarks>
				///
				/// <returns>	true if it succeeds, false if it fails. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual bool Init() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Get posterior distribution of latent variables /f$\mathbf{X}/f$. </summary>
				///
				/// <remarks>	Hmetal T, 09/12/2019. </remarks>
				///
				/// <param name="index">	Index of selected inputs. </param>
				/// <param name="mx">   	[in,out] The mean. </param>
				/// <param name="vx">   	[in,out] The variance. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void PosteriorLatents(af::array& mx, af::array& vx);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param) override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters for each optimization iteration. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the parameters. </summary>
				///
				/// <remarks>	, 26.06.2018. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void UpdateParameters() override;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets fixation for hyperparameters. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixKernelParameters(bool isfixed);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Set fixation for inducing inputs. </summary>
				///
				/// <remarks>	Hmetal T, 16/12/2019. </remarks>
				///
				/// <param name="isfixed">	True if isfixed. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void FixInducing(bool isfixed);

				void FixLatents(bool isFixed);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets prior mean gradient. </summary>
				///
				/// <remarks>	Hmetal T, 31/08/2020. </remarks>
				///
				/// <returns>	The mean gradient. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetMeanGradient();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets prior variance gradient. </summary>
				///
				/// <remarks>	Hmetal T, 31/08/2020. </remarks>
				///
				/// <returns>	The variable gradient. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				af::array GetVarGradient();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the prior. </summary>
				///
				/// <remarks>	Hmetal T, 01/09/2020. </remarks>
				///
				/// <param name="mean">	The mean. </param>
				/// <param name="var"> 	The variable. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetPrior(const af::array mean, const af::array var);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the cavity prior. </summary>
				///
				/// <remarks>	Hmetal T, 20/04/2021. </remarks>
				///
				/// <param name="meanCav">	The mean cav. </param>
				/// <param name="varCav"> 	The variable cav. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetPriorCavity(const af::array meanCav, const af::array varCav);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets latent gradient. </summary>
				///
				/// <remarks>	Hmetal T, 20/04/2021. </remarks>
				///
				/// <param name="dmParent">	The dm parent. </param>
				/// <param name="dvParent">	The dv parent. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetLatentGradient(const af::array& dmParent, const af::array& dvParent);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the latent cavity gradient. </summary>
				///
				/// <remarks>	Hmetal T, 20/04/2021. </remarks>
				///
				/// <param name="dmParent">	The dm parent. </param>
				/// <param name="dvParent">	The dv parent. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetLatentGradientCavity(const af::array& dmParent, const af::array& dvParent);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets latent dimension. </summary>
				///
				/// <remarks>	Hmetal T, 01/09/2020. </remarks>
				///
				/// <returns>	The latent dimension. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				int GetLatentDimension();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets a back-constraint. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="constraint">	The constraint. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetBackConstraint(IBackconstraint<Scalar>* constraint);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the back-constraint. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	Null if it fails, else the back constraint. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				IBackconstraint<Scalar>* GetBackConstraint();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the syles. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <param name="styles">	[in,out] If non-null, the styles. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void SetStyles(std::map<std::string, Style<Scalar>>* styles);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Adds a style. </summary>
				///
				/// <remarks>	Hmetal T, 02/11/2020. </remarks>
				///
				/// <param name="styles">	The styles. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				void AddStyle(Style<Scalar> style);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the styles. </summary>
				///
				/// <remarks>	Hmetal T, 25/09/2020. </remarks>
				///
				/// <returns>	Null if it fails, else the styles. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				std::map<std::string, Style<Scalar>>* GetStyles();
				
			protected:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Posterior gradient of latent inputs /f$\mathbf{X}/f$. </summary>
				///
				/// <remarks>	Hmetal T, 09/12/2019. </remarks>
				///
				/// <param name="dmx">	[in,out] The gradient of the mean. </param>
				/// <param name="dvx">	[in,out] The the gradient of the variance. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array PosteriorGradientLatents(const af::array& dmx, const af::array& dvx);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Updates the internal parameters. </summary>
				///
				/// <remarks>	Hmetal T, 23/03/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void UpdateParametersInternal();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Latent gradient. </summary>
				///
				/// <remarks>	Hmetal T, 20/04/2021. </remarks>
				///
				/// <returns>	An af::array. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array LatentGradient(const af::array& dm, const af::array& dv);
				
				int iq;			//!< latent dimension

				Scalar dPriorMean;		//!< prior mean
				Scalar dPriorVariance;	//!< prior variance
				Scalar dPriorX1;		//!< prior /f$x_1/f$
				Scalar dPriorX2;		//!< prior /f$x_2/f$

				af::array afFactorX1;		//!< natural parameter factor 1 for latent variable
				af::array afFactorX2;		//!< natural parameter factor 2 for latent variable
				af::array afPosteriorX1;	//!< posterior natural parameter 1 for latent variable
				af::array afPosteriorX2;	//!< posterior natural parameter 2 for latent variable

				af::array afPriorMean;		//!< prior mean for hiersrchy mode
				af::array afPriorVariance;	//!< prior variance for hiersrchy mode
				af::array afPriorMeanCav;		//!< prior mean for hiersrchy mode
				af::array afPriorVarianceCav;	//!< prior variance for hiersrchy mode
				af::array afPriorX1;		//!< prior /f$x_1/f$
				af::array afPriorX2;		//!< prior /f$x_2/f$
				af::array afPriorX1Cav;		//!< prior /f$x_1/f$
				af::array afPriorX2Cav;		//!< prior /f$x_2/f$
				af::array afGradMean;		//!< prior mean gradient for hiersrchy mode
				af::array afGradVariance;	//!< prior variance gradient for hiersrchy mode
				af::array afGradMeanCav;		//!< prior mean gradient for hiersrchy mode
				af::array afGradVarianceCav;	//!< prior variance gradient for hiersrchy mode

				af::array afLatentGradientX;	//!< top down gradient

				XInit eEmMethod;

				IBackconstraint<Scalar>* backConst;				//!< back-constraint
				std::map<std::string, Style<Scalar>>* mStyles;	//!< style variable

				bool bIsLatetsFixed;

			private:
				friend class boost::serialization::access;
				friend class KBR<Scalar>;
				friend class PTC<Scalar>;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<GPBaseModel<Scalar>>(*this);
					ar& boost::serialization::base_object<GPNode<Scalar>>(*this);

					/*ar& boost::serialization::make_nvp("GPBaseModel", boost::serialization::base_object<GPBaseModel<Scalar>>(*this));
					ar& boost::serialization::make_nvp("GPNode", boost::serialization::base_object<GPNode<Scalar>>(*this));*/

					ar.register_type<KBR<Scalar>>();
					ar.register_type<PTC<Scalar>>();

					ar& BOOST_SERIALIZATION_NVP(iq);
					ar& BOOST_SERIALIZATION_NVP(dPriorMean);
					ar& BOOST_SERIALIZATION_NVP(dPriorVariance);
					ar& BOOST_SERIALIZATION_NVP(afFactorX1);
					ar& BOOST_SERIALIZATION_NVP(afFactorX2);
					ar& BOOST_SERIALIZATION_NVP(eEmMethod);
					ar& BOOST_SERIALIZATION_NVP(afPriorMean);
					ar& BOOST_SERIALIZATION_NVP(afPriorVariance);
					ar& BOOST_SERIALIZATION_NVP(afPriorMeanCav);
					ar& BOOST_SERIALIZATION_NVP(afPriorVarianceCav);
					ar& BOOST_SERIALIZATION_NVP(afPriorX1);
					ar& BOOST_SERIALIZATION_NVP(afPriorX2);
					ar& BOOST_SERIALIZATION_NVP(backConst);
					ar& BOOST_SERIALIZATION_NVP(mStyles);
					ar& BOOST_SERIALIZATION_NVP(bIsLatetsFixed);
				}

			};
		}
	}
}
