/**
File:		MachineLearning/GPModels/Backconstraints/FgIBackconstraints.h

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/CommonUtil.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			enum class BackConstType
			{
				kbr,		// Kernel-based regression back constraint function.
				mlp,		// Multi-layer perceptron back constraint function.
				ptc,		// periodic topological constraint
				halfPtc,    // half circle periodic topological constraint
				ptcLinear,
				ptcHardCircle
			};

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	
			/// 	Abstract class for back-constraints, a kind of prior knowledge to force topological positions
			/// 	of uncertain latent inputs. All types of back-constraints will be derived from this class.			
			/// </summary>
			///
			/// <remarks>	HmetalT, 26.10.2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP IBackconstraint
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Constructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="type">	The type. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				IBackconstraint(BackConstType type);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~IBackconstraint();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes this object. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="Y">	An af::array to process. </param>
				/// <param name="X">	An af::array to process. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void Init(const af::array& Y, const af::array& X, const af::array& segments) = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of to be optimized parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters() = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param) = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters() = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets constraint x coordinates. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The constraint x coordinate. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetConstraintX() = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Back-constraint gradient. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="gX">	The gradient of \f$\mathbf{X}\f$. </param>
				///
				/// <returns>	The gradient of the back-constraint. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array BackconstraintGradient(const af::array& gX) = 0;

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the back-constraint type. </summary>
				///
				/// <remarks>	Hmetal T, 15/09/2020. </remarks>
				///
				/// <returns>	The type. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				BackConstType GetType();

			protected:
				int iN;				//!< dataset length
				int iq;				//!< latent dimension

				af::dtype m_dType;	//!< floating point precision flag for af::array

			private:
				BackConstType backType;

				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& BOOST_SERIALIZATION_NVP(iN);
					ar& BOOST_SERIALIZATION_NVP(iq);
					ar& BOOST_SERIALIZATION_NVP(backType);
				}
			};
		}
	}
}
/** @example AEP_SGPLVM_Backconstraints_Examples.cpp */