﻿/**
File:		MachineLearning/Embed/FgLLE.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <MachineLearning/IEmbed.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Isomap. </summary>
		///
		/// <remarks>
		/// 	<para>
		/// 		Isomap is one representative of isometric mapping methods, and extends metric 
		/// 		multidimensional scaling (MDS) by incorporating the geodesic distances imposed by 
		/// 		a weighted graph. To be specific, the classical scaling of metric MDS performs 
		/// 		low-dimensional embedding based on the pairwise distance between data points, 
		/// 		which is generally measured using straight-line Euclidean distance. Isomap is 
		/// 		distinguished by its use of the geodesic distance induced by a neighborhood graph 
		/// 		embedded in the classical scaling. This is done to incorporate manifold structure 
		/// 		in the resulting embedding. Isomap defines the geodesic distance to be the sum of 
		/// 		edge weights along the shortest path between two nodes (computed using Dijkstra's 
		/// 		algorithm, for example). The top n eigenvectors of the geodesic distance matrix, 
		/// 		represent the coordinates in the new n-dimensional Euclidean space.
		/// 		
		/// 		Translation of Tenenbaum, de Silva, and Langford' Matlab code.
		///		</para>
		///		<para>
		///			Algorithm:
		///			<list type="number">
		///				<item>
		///					Determine the neighbors of each point.
		///					<list type="bullet">
		///					<item>
		///					All points in some fixed radius.
		///					</item>
		///					<item>
		///					K nearest neighbors.
		///					</item>
		///					</list>
		///				</item>
		///				<item>
		///					Construct a neighborhood graph.
		///					<list type="bullet">
		///					<item>
		///					Each point is connected to other if it is a K nearest neighbor.
		///					</item>
		///					<item>
		///					Edge length equal to Euclidean distance.
		///					</item>
		///					</list>
		///				</item>
		///				<item>
		///					Compute shortest path between two nodes.
		///					<list type="bullet">
		///					<item>
		///					Dijkstra's algorithm
		///					</item>
		///					<item>
		///					Floyd–Warshall algorithm
		///					</item>
		///					</list>
		///				</item>
		///				<item>
		///					Compute lower - dimensional embedding.
		///					<list type="bullet">
		///					<item>
		///					Multidimensional scaling
		///					</item>
		///					</list>
		///				</item>
		///			</list>
		///		</para>
		///	</remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		class NE_IMPEXP Isomap : public IEmbed
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Constructor. </summary>
			///
			/// <remarks>	Hmetal T, 17.04.2017. </remarks>
			///
			/// <param name="neighbours">	The number of neighbours to compute. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			Isomap(int numNeighbours);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	Hmetal T, 17.04.2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			~Isomap();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Solves. </summary>
			///
			/// <remarks>	Hmetal T, 11.04.2017. </remarks>
			///
			/// <param name="M">	[in,out] N by D data matrix. </param>
			/// <param name="q">	Latent dimension. </param>
			///
			/// <returns>	Latent points. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			af::array Compute(af::array& M, int q);

		private:
			struct Xstruct
			{
				af::array index;
				std::vector<af::array> coords;
			};

			Xstruct isomap(af::array& D, int n_size, int q);
			af::array L2_distance(af::array& a, af::array& b, bool df);

			int _numNeighbours;
		};
	}
}
