﻿/**
File:		MachineLearning/Embed/FgLLE.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <MachineLearning/IEmbed.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Locally-Linear Embedding. </summary>
		///
		/// <remarks>
		/// 	<para>
		/// 		Locally-Linear Embedding (LLE) was presented at approximately the same time as Isomap. 
		/// 		It has several advantages over Isomap, including faster optimization when implemented 
		/// 		to take advantage of sparse matrix algorithms, and better results with many problems. 
		/// 		LLE also begins by finding a set of the nearest neighbors of each point. It then computes
		/// 		a set of weights for each point that best describe the point as a linear combination of 
		/// 		its neighbors. Finally, it uses an eigenvector-based optimization technique to find the 
		/// 		low-dimensional embedding of points, such that each point is still described with the 
		/// 		same linear combination of its neighbors. LLE tends to handle non-uniform sample densities 
		/// 		poorly because there is no fixed unit to prevent the weights from drifting as various 
		/// 		regions differ in sample densities. LLE has no internal model.
		///		</para>
		///		<para>
		///			LLE computes the barycentric coordinates of a point \f$\mathbf{x}_i\f$ based on its 
		///			neighbors \f$\mathbf{x}_j\f$.The original point is reconstructed by a linear combination, 
		///			given by the weight matrix \f$\mathbf{W}_{ij}\f$, of its neighbors. The reconstruction 
		///			error is given by the cost function \f$E(\mathbf{W})\f$,
		///
		///			\f[ E(\mathbf{W}) = \sum_i | \mathbf{x}_i − \sum_j \mathbf{W}_{ij} \mathbf{x}_j |^2 .\f]
		///
		///			The weights \f$\mathbf{W}_{ij}\f$ refer to the amount of contribution the point 
		///			\f$\mathbf{x}_j\f$ has while reconstructing the point \f$\mathbf{x}_i\f$.The cost 
		///			function is minimized under two constraints:
		///				
		///			<list type="number">
		///			<item>
		///				Each data point \f$\mathbf{x}_i\f$ is reconstructed only from its neighbors, 
		///				thus enforcing \f$\mathbf{W}_{ij}\f$ to be zero if point \f$\mathbf{x}_j\f$ 
		///				is not a neighbor of the point \f$\mathbf{x}_i\f$ and
		///			</item>
		///			<item>
		///				The sum of every row of the weight matrix equals 1,
		///				\f[\sum_j \mathbf{W}_{ij} = 1.\f]
		///			</item>
		///			</list>
		///
		///			The original data points are collected in a D dimensional space and the goal of the 
		///			algorithm is to reduce the dimensionality to d such that \f$D >> d\f$.The same weights 
		///			\f$\mathbf{W}_{ij}\f$ that reconstructs the \f$i\f$th data point in the \f$D\f$-dimensional 
		///			space will be used to reconstruct the same point in the lower \f$d\f$-dimensional space.
		///			A neighborhood preserving map is created based on this idea.Each point \f$\mathbf{x}_i\f$ 
		///			in the \f$D\f$-dimensional space is mapped onto a point \f$\mathbf{Y}_i\f$ in the
		///			\f$d\f$-dimensional space by minimizing the cost function,
		///
		///			\f[C(\mathbf{Y}) = \sum_i | \mathbf{Y}_i − \sum_j \mathbf{W}_{ij} \mathbf{Y}_j |^2.\f]
		///
		///			In this cost function, unlike the previous one, the weights \f$\mathbf{W}_{ij}\f$ are 
		///			kept fixed and the minimization is done on the points \f$\mathbf{Y}_i\f$ to optimize 
		///			the coordinates.This minimization problem can be solved by solving a sparse 
		///			\f$N \times N\f$ eigen value problem(\f$N\f$ being the number of data points), 
		///			whose bottom d nonzero eigen vectors provide an orthogonal set of coordinates.
		///			Generally the data points are reconstructed from \f$K\f$ nearest neighbors, as measured 
		///			by Euclidean distance.For such an implementation the algorithm has only one free 
		///			parameter \f$K\f$, which can be chosen by cross validation.
		///		</para>
		///	</remarks>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		class NE_IMPEXP LLE : public IEmbed
		{
		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Constructor. </summary>
			///
			/// <remarks>	Hmetal T, 11.04.2017. </remarks>
			///
			/// <param name="numNeighbours">	Number of neighbours. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			LLE(int numNeighbours);

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Destructor. </summary>
			///
			/// <remarks>	Hmetal T, 11.04.2017. </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			~LLE();

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Solves. </summary>
			///
			/// <remarks>	Hmetal T, 11.04.2017. </remarks>
			///
			/// <param name="M">	[in,out] N by D data matrix. </param>
			/// <param name="q">	Latent dimension. </param>
			///
			/// <returns>	Latent points. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			af::array Compute(af::array& M, int q);

		private:
			int _numNeighbours;
		};
	}
}
