﻿/**
File:		MachineLearning/GPModels/SparseGPModels/PowerEP/FgPEPSparseGPLayer2nd.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgSparseGPBaseLayer.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			namespace PowerEP
			{
				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sparse GP layer. </summary>
				///
				/// <remarks>
				/// 	Holds all variables for FITC approximation and PEP. Defines a subset of X. Selects k 
				/// 	inducing inputs Xu, computes the subset kernel matrix Kuu and its inverse.
				/// 			
				///		For more information see,
				///		https://pdfs.semanticscholar.org/99f9/3283e415ae21bd42a90031cd3972f3bfbc9d.pdf
				///				
				/// 	Hmetal T, 05/05/2018. 
				/// </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				template<typename Scalar>
				class NE_IMPEXP SGPLayer : public SparseGPBaseLayer<Scalar>
				{
				public:
					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Constructor. </summary>
					///
					/// <remarks>	, 15.05.2018. </remarks>
					///
					/// <param name="numPoints"> 	Number of points. </param>
					/// <param name="numPseudos">	Number of pseudo inputs. </param>
					/// <param name="outputDim"> 	The output dimension. </param>
					/// <param name="inputDim">  	The input dimension. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					SGPLayer(int numPoints, int numPseudos, int outputDim, int inputDim);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Destructor. </summary>
					///
					/// <remarks>	, 15.05.2018. </remarks>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					virtual ~SGPLayer();

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Forward prediction through cavity. </summary>
					///
					/// <remarks>	
					///		Computes new mean m^{\n}_{f} and covariance V^{\n}_{ff} function. From cavity
					///		distribution.
					///								
					/// 	, 16.05.2018. 
					/// </remarks>
					///
					/// <param name="mout"> 	[in,out] The m^{\n}_{f}. </param>
					/// <param name="vout"> 	[in,out] The V^{\n}_{ff}. </param>
					/// <param name="n">		[in,out] indexes of points to be removed from distribution. </param>
					/// <param name="mx">   	[in,out] The inputs mx. </param>
					/// <param name="vx">   	[in,out] (Optional) If non-null, the variances vx. </param>
					/// <param name="alpha">	(Optional) the alpha, weighning for alpha-divergence. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					void ForwardPredictionCavity(af::array& mout, af::array& vout, af::array& n, af::array& mx, af::array* vx = nullptr, Scalar alpha = 1.0);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Prediction of posterior function values. </summary>
					///
					/// <remarks>	, 12.06.2018. </remarks>
					///
					/// <param name="testX">	[in,out] The test inputs. </param>
					/// <param name="mf">   	[in,out] mean function values. </param>
					/// <param name="vf">   	[in,out] covariance function values. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					//virtual void ForwardPredictionPost(af::array& mout, af::array& vout, af::array* mx, af::array* vx = nullptr) override;

					void BackpropGradientsReg(af::array& m, af::array& v, af::array& dlogZ_dm, af::array& dlogZ_dv, af::array& x, std::map<std::string,
						af::array>& grad_hyper, std::map<std::string, af::array>& grad_cav, Scalar alpha = 1.0);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Update step of PowerEP. </summary>
					///
					/// <remarks>	
					/// 			
					/// 	Hmetal T, 08/06/2018. 
					/// </remarks>
					///
					/// <param name="n">	   	[in,out] The indexes to compute. </param>
					/// <param name="grad_cav">	[in,out] The gradient of the cavity functions m_f^{\n} and V_{ff}^{\m}. </param>
					/// <param name="alpha">   	The alpha. </param>
					/// <param name="decay">   	(Optional) the decay. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					void UpdateFactor(af::array& n, std::map<std::string, af::array> grad_cav, Scalar alpha, Scalar decay = 0);

					virtual void UpdateParameters() override;

					virtual void InitParameters(af::array* X = nullptr) override;

				protected:
					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Default constructor. </summary>
					///
					/// <remarks>	Hmetal T, 02/07/2018. </remarks>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					SGPLayer() { }

					void ForwardPredictionDeterministicPost(af::array& mx, af::array* mout, af::array* vout);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Forward prediction through deterministic cavity. </summary>
					///
					/// <remarks>	
					///		Projection step. Computation of cavity posterior mean and covariance function,
					///		
					///			m^{\n}_{f} = K_{fu}K_{uu}^{−1}T^{\n, −1}_{2, u}T^{\n}_{1, u},
					///			
					///			V{\n}_{ff} = K_{ff} − Q_{ff} + K_{fu}K^{−1}_{uu}T^{\n, −1}_{2, u}K^{−1}_{uu}K_{uf}.
					///								
					/// 	, 16.05.2018. 
					/// </remarks>
					///
					/// <param name="mout"> 	[in,out] The m^{\n}_{f}. </param>
					/// <param name="vout"> 	[in,out] The V^{\n}_{ff}. </param>
					/// <param name="idx">		[in,out] indexes of points to be removed from distribution. </param>
					/// <param name="mx">   	[in,out] The inputs mx. </param>
					/// <param name="alpha">	(Optional) the alpha, weighning for alpha-divergence. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					void ForwardPredictionDeterministicCavity(af::array& mout, af::array& vout, af::array& idx, af::array& mx, Scalar alpha = 1.0);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Forward prediction through random cavity. </summary>
					///
					/// <remarks>	, 16.05.2018. </remarks>
					///
					/// <param name="mout"> 	[in,out] The m^{\n}_{f}. </param>
					/// <param name="vout"> 	[in,out] The V^{\n}_{ff}. </param>
					/// <param name="idx">		[in,out] indexes of points to be removed from distribution. </param>
					/// <param name="mx">   	[in,out] The inputs mx. </param>
					/// <param name="vx">   	[in,out] (Optional) If non-null, the variances vx. </param>
					/// <param name="alpha">	(Optional) the alpha, weighning for alpha-divergence. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					void ForwardPredictionRandomCavity(af::array& mout, af::array& vout, af::array& idx, af::array& mx, af::array& vx, Scalar alpha = 1.0);

					////////////////////////////////////////////////////////////////////////////////////////////////////
					/// <summary>	Calculates the cavity. </summary>
					///
					/// <remarks>
					/// 	Deletion step: The cavity for data point n, q^{\n}(f) ∝ q^∗(f)/t^α_n(u), has a similar
					/// 	form to the posterior, but the natural parameters are modified by the deletion,
					/// 	
					/// 		T^\n_{1, u} = T_{1, u} − αT_{1, n}
					/// 	
					/// 			and
					/// 	
					/// 		T^\n_{2, u} = T_{2, u} − αT_{2, n},
					/// 	
					/// 	for yielding new mean muhat and covariance function Suhat.
					/// 	
					/// 	, 16.05.2018.
					/// </remarks>
					///
					/// <param name="idx">	 	[in,out] indexes of points to be removed from distribution. </param>
					/// <param name="muhat"> 	[in,out] The muhat. </param>
					/// <param name="Suhat"> 	[in,out] The Suhat. </param>
					/// <param name="T1uHat">	[in,out] The cavity natural parameter T^\n_{1, u}. </param>
					/// <param name="T2uHat">	[in,out] The cavity natural parameter T^\n_{2, u}. </param>
					/// <param name="alpha"> 	(Optional) the alpha, weighning for alpha-divergence. </param>
					////////////////////////////////////////////////////////////////////////////////////////////////////
					void ComputeCavity(af::array& idx, af::array& muhat, af::array& Suhat, af::array& T1uHat, af::array& T2uHat, Scalar alpha = 1.0f);

					// PEP variables
					af::array gamma;
					af::array beta;

					//// factor variables
					//af::array T1;
					//af::array T2;

					// Cavity temp variables
					af::array afMuHat;			// mean^{\n} of q(u)
					af::array afSuHat;			// covariance^{\n} of q(u)
					af::array afInvSuHat;		// T_{2,u}^{\n}
					af::array afInvSuMuHat;		// T_{1,u}^{\n}

				private:
					friend class boost::serialization::access;

					template<class Archive>
					void serialize(Archive& ar, unsigned int version)
					{
						ar& boost::serialization::base_object<SparseGPBaseLayer<Scalar>>(*this);
						//ar& boost::serialization::make_nvp("SparseGPBaseLayer", boost::serialization::base_object<SparseGPBaseLayer<Scalar>>(*this));
						ar& BOOST_SERIALIZATION_NVP(gamma);
						ar& BOOST_SERIALIZATION_NVP(beta);
					}
				};
			}
		}
	}
}
