/**
File:		MachineLearning/GPModels/Backconstraints/FgPeriodicTopologicalConstraint.h

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#pragma once

#include <MachineLearning/FgIBackconstraint.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		namespace GPModels
		{
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	
			/// 	Back-constraints via Periodic topological constraint.			
			/// </summary>
			///
			/// <remarks>	
			///		<para>
			///			For PTC the content vectors are restricted to lie on a unit circle. The low-dimensional 
			///			coordinates are parameterized by a phase parameter \f$\phi_t\f$, such that 
			///			\f$\mathbf{x}_t = [\cos \phi_t, \sin\phi_t]^T\f$. Phase is linear as a function of 
			///			time, parameterized by offset \f$\phi_0\f$ and step-size \f$\Delta\phi: \phi_t = \phi_0 
			///			+ t\Delta\phi\f$. Each sequence is then parameterized only by \f$\phi_0\f$ and   
			///			\f$\Delta\phi\f$. The step-size accounts for the different frequencies of different gaits.
			///		</para>
			///		
			///		<para>
			///		  Reference:
			///		  <list type="bullet">
			///			<item>
			///		    	  <description><a href="https://www.cs.princeton.edu/courses/archive/spring11/cos598A/pdfs/Wang07.pdf" target="_blank">
			///					Wang, Jack M., Fleet, David J., Hertzmann, Aaron. (2007). "Multifactor 
			///					Gaussian Process Models for Style-Content Separation".
			///					Proceedings of the 24th international conference on Machine learning - 
			///					ICML, New York, NY, USA </a>
			///		       </description>
			///		    </item>
			///		</para>
			///								
			/// 	HmetalT, 26.10.2017. 
			/// </remarks>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			template<typename Scalar>
			class NE_IMPEXP PTC : public IBackconstraint<Scalar>
			{
			public:

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Default constructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				PTC();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Destructor. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual ~PTC();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Initializes this object. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="Y">	An af::array to process. </param>
				/// <param name="X">	An af::array to process. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void Init(const af::array& Y, const af::array& X, const af::array& segments);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets number of to be optimized parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The number parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual int GetNumParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Sets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="param">	The parameter. </param>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual void SetParameters(const af::array& param);

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets the parameters. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The parameters. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetParameters();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Gets constraint x coordinates. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <returns>	The constraint x coordinate. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array GetConstraintX();

				////////////////////////////////////////////////////////////////////////////////////////////////////
				/// <summary>	Back-constraint gradient. </summary>
				///
				/// <remarks>	Hmetal T, 17/09/2020. </remarks>
				///
				/// <param name="gX">	The gradient of \f$\mathbf{X}\f$. </param>
				///
				/// <returns>	The gradient of the back-constraint. </returns>
				////////////////////////////////////////////////////////////////////////////////////////////////////
				virtual af::array BackconstraintGradient(const af::array& gX);

			protected:
				af::array afA;
				std::vector<int> pSegments;

				int iNumSegments;

			private:
				friend class boost::serialization::access;

				template<class Archive>
				void serialize(Archive& ar, unsigned int version)
				{
					ar& boost::serialization::base_object<IBackconstraint<Scalar>>(*this);

					ar& BOOST_SERIALIZATION_NVP(afA);
					ar& BOOST_SERIALIZATION_NVP(pSegments);
					ar& BOOST_SERIALIZATION_NVP(iNumSegments);
				}
			};
		}
	}
}
