/**afaf::array
File:		MachineLearning/Optimization/Base/IIOptimizationMethod.h

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#pragma once

#include <NeMachineLearningLib.h>
#include <MachineLearning/CommonUtil.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		enum OptimizerType
		{
			SCG,
			L_BFGS_B,
			L_BFGS,
			ADAM,
			ADAMAX,
			NADAM
		};

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Common interface for function optimization methods. </summary>
		///
		/// <remarks>	Hmetal T, 16.03.2017. </remarks>
		///
		/// <seealso cref="BoundedBroydenFletcherGoldfarbShanno"/>
		/// <seealso cref="BroydenFletcherGoldfarbShanno"/>
		/// <seealso cref="ConjugateGradient"/>
		/// <seealso cref="ResilientBackpropagation"/>
		/// <seealso cref="GoldfarbIdnani"/>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename Scalar>
		class IOptimizationMethod
		{

		public:

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Gets the number of variables (free parameters)
			/// 	in the optimization problem.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>	The number of parameters. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual int GetNumberOfVariables() = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Gets the current solution found, the values of the parameters which optimizes the
			/// 	function.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>	An af::array. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual af::array GetSolution() = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets a solution. </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <param name="parameter1">	The first parameter. </param>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual void SetSolution(af::array& x) = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>	Gets the output of the function at the current <see cref="Solution"/>. </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>	A double. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual Scalar GetValue() = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the minimum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Minimize(int* cycle = nullptr) = 0;

			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Finds the maximum value of a function. The solution vector will be made available at the
			/// 	<see cref="Solution"/> property.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>
			/// 	Returns <c>true</c> if the method converged to a <see cref="Solution"/>. In this case,
			/// 	the found value will also be available at the <see cref="Value"/>
			/// 	property.
			/// </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual bool Maximize(int* cycle = nullptr) = 0;

			~IOptimizationMethod() {};
		};

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// <summary>	Common interface for function optimization methods. </summary>
		///
		/// <remarks>	Hmetal T, 16.03.2017. </remarks>
		///
		/// <seealso cref="BoundedBroydenFletcherGoldfarbShanno"/>
		/// <seealso cref="BroydenFletcherGoldfarbShanno"/>
		/// <seealso cref="ConjugateGradient"/>
		/// <seealso cref="ResilientBackpropagation"/>
		/// <seealso cref="GoldfarbIdnani"/>
		////////////////////////////////////////////////////////////////////////////////////////////////////
		template<typename TCode>
		class  OptimizationMethod
		{
		public:
			////////////////////////////////////////////////////////////////////////////////////////////////////
			/// <summary>
			/// 	Get the exit code returned in the last call to the
			/// 	<see cref="IOptimizationMethod.Maximize()"/> or
			/// 	<see cref="IOptimizationMethod.Minimize()"/> methods.
			/// </summary>
			///
			/// <remarks>	Hmetal T, 16.03.2017. </remarks>
			///
			/// <returns>	A TCode. </returns>
			////////////////////////////////////////////////////////////////////////////////////////////////////
			virtual TCode GetStatus() = 0;
		};
	}
}