/*
  Copyright (c) 2012, Chris McClanahan
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  * Neither the name of the copyright holder nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/**
 af_cv: Misc ArrayFire 2.0 OpenCV utility fuctions
 */

#include <NeMachineLearningPCH.h>
#include <MachineLearning/FgAfCv.h>
#include <af/defines.h>
#include <iostream>
#include <iomanip>
#include <stdexcept>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		// mem layout for gpu
		void AfCv::MatToArray(const cv::Mat& input, af::array& output)
		{
			const unsigned w = input.cols;
			const unsigned h = input.rows;
			const unsigned channels = input.channels();

			if (channels == 1) 
			{
				if (input.type() == CV_32F)
				{
					output = af::array(f32);
					output = af::array(w, h, input.ptr<float>(0)).T();
				}		
				else
				{
					output = af::array(w, h, input.ptr<double>(0)).as(f64).T();
				}	
			}
			else 
			{
				af::array input_;
				if (input.type() == CV_32F || input.type() == CV_32FC2 || input.type() == CV_32FC3)
					input_ = af::array(w * channels, h, input.ptr<float>(0));
				else
					input_ = af::array(w * channels, h, input.ptr<double>(0)).as(f64);

				af::array ind = af::array(af::seq(channels - 1, w * channels - 1, channels));

				if (input.type() == CV_32F || input.type() == CV_32FC2 || input.type() == CV_32FC3)
					output = af::array(h, w, channels);
				else
					output = af::array(h, w, channels, f64);

				for (int ii = 0; ii < channels; ii++) 
				{
					output(af::span, af::span, channels - ii - 1) = input_(ind - ii, af::span).T();
				}
			}
		}
		
		af::array AfCv::MatToArray(const cv::Mat& input) 
		{
			af::array output;
			if (input.empty()) return output;

			MatToArray(input, output);
			return output;
		}

		void AfCv::MatToArray(const std::vector<cv::Mat>& input, af::array& output)
		{
			try 
			{
				int d = input[0].dims;
				int h = input[0].rows;
				int w = input[0].cols;

				if (input[0].type() == CV_32F)
					output = af::array(h, w, input.size());
				else
					output = af::array(h, w, input.size(), f64);

				for (unsigned i = 0; i < input.size(); i++) 
				{
					af::array tmp = MatToArray(input[i]);
					switch (d) 
					{
					case 1: output(af::span, i) = tmp;
						break;
					case 2:
						if (!tmp.isempty())
							output(af::span, af::span, i) = tmp;
						else
							output = af::array();
						break;
					default: throw std::runtime_error(std::string("Only 1 and 2 dimensions supported"));
					}
				}
			}
			catch (const af::exception& ex) 
			{
				throw std::runtime_error(std::string("mat to array error "));
			}
		}

		cv::Mat AfCv::ArrayToMat(const af::array& input, int type) {
			cv::Mat output;
			
			const int channels = input.dims(2);
			int ndims = input.numdims();

			af::array input_;

			if (channels == 1) 
				input_ = input.T();
			else 
				input_ = af::reorder(input.T(), 2, 0, 1); // This converts the 3rd dimension to be the first dimension

			if (type == CV_32F)
			{
				if (ndims > 1)
					output = cv::Mat(input_.dims(ndims - 1), input_.dims(ndims - 2), CV_MAKETYPE(type, channels), input_.host<float>());
				else
					output = cv::Mat(input_.dims(1), input_.dims(0), CV_MAKETYPE(type, channels), input_.host<float>());
			}
			else if (type == CV_32S) 
			{
				if (ndims > 1)
					output = cv::Mat(input_.dims(ndims - 1), input_.dims(ndims - 2), CV_MAKETYPE(type, channels), input_.host<int>());
				else
					output = cv::Mat(input_.dims(1), input_.dims(0), CV_MAKETYPE(type, channels), input_.host<int>());
			}
			else if (type == CV_64F)
			{
				if (ndims > 1)
					output = cv::Mat(input_.dims(ndims - 1), input_.dims(ndims - 2), CV_MAKETYPE(type, channels), input_.host<double>());
				else
					output = cv::Mat(input_.dims(1), input_.dims(0), CV_MAKETYPE(type, channels), input_.host<double>());
			}
			else if (type == CV_8U)
			{
				if (ndims > 1)
					output = cv::Mat(input_.dims(ndims - 1), input_.dims(ndims - 2), CV_MAKETYPE(type, channels), input_.host<uchar>());
				else
					output = cv::Mat(input_.dims(1), input_.dims(0), CV_MAKETYPE(type, channels), input_.host<uchar>());
			}
			else
			{
				throw std::runtime_error(std::string("array to mat error "));
			}

			return output;
		}


		// Mat type info
		std::string AfCv::get_mat_type(const cv::Mat& input) { return get_mat_type(input.type()); }
		std::string AfCv::get_mat_type(int input) {
			int numImgTypes = 35; // 7 base types, with five channel options each (none or C1, ..., C4)
			int enum_ints[] = { CV_8U,  CV_8UC1,  CV_8UC2,  CV_8UC3,  CV_8UC4,
				CV_8S,  CV_8SC1,  CV_8SC2,  CV_8SC3,  CV_8SC4,
				CV_16U, CV_16UC1, CV_16UC2, CV_16UC3, CV_16UC4,
				CV_16S, CV_16SC1, CV_16SC2, CV_16SC3, CV_16SC4,
				CV_32S, CV_32SC1, CV_32SC2, CV_32SC3, CV_32SC4,
				CV_32F, CV_32FC1, CV_32FC2, CV_32FC3, CV_32FC4,
				CV_64F, CV_64FC1, CV_64FC2, CV_64FC3, CV_64FC4
			};
			std::string enum_strings[] = { "CV_8U",  "CV_8UC1",  "CV_8UC2",  "CV_8UC3",  "CV_8UC4",
				"CV_8S",  "CV_8SC1",  "CV_8SC2",  "CV_8SC3",  "CV_8SC4",
				"CV_16U", "CV_16UC1", "CV_16UC2", "CV_16UC3", "CV_16UC4",
				"CV_16S", "CV_16SC1", "CV_16SC2", "CV_16SC3", "CV_16SC4",
				"CV_32S", "CV_32SC1", "CV_32SC2", "CV_32SC3", "CV_32SC4",
				"CV_32F", "CV_32FC1", "CV_32FC2", "CV_32FC3", "CV_32FC4",
				"CV_64F", "CV_64FC1", "CV_64FC2", "CV_64FC3", "CV_64FC4"
			};
			for (int i = 0; i < numImgTypes; i++) {
				if (input == enum_ints[i]) { return enum_strings[i]; }
			}
			return "unknown image type";
		}
	}
}

