/**
File:		MachineLearning/Models/GPModels/FgSparseDeepGPBaseModel.cpp

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#include <NeMachineLearningPCH.h>
#include <MachineLearning/FgGPLVMBaseModel.h>

namespace NeuralEngine::MachineLearning::GPModels
{
	template class GPLVMBaseModel<float>;
	template class GPLVMBaseModel<double>;

	template class Style<float>;
	template class Style<double>;


	template<typename Scalar>
	Style<Scalar>::Style(std::string name)
		: sName(name), afStyleIdx(), afStyleInducingIdx(), afStyle(), afInducingStyle(), mSubStyles(), m_dType(CommonUtil<Scalar>::CheckDType())
	{
	}

	template<typename Scalar>
	Style<Scalar>::~Style()
	{
	}

	template<typename Scalar>
	void Style<Scalar>::AddSubstyle(std::string name, int numFrames)
	{
		auto it = mSubStyles.find(name);
		if (it == mSubStyles.end())
		{
			mSubStyles.insert(std::make_pair(name, mSubStyles.size()));
			it = mSubStyles.find(name);

			afStyle = af::identity(mSubStyles.size(), mSubStyles.size(), m_dType);
		}

		afStyleIdx = CommonUtil<Scalar>::Join(afStyleIdx, af::constant(it->second, numFrames));

		afInducingStyle = afStyle.copy();
	}

	template<typename Scalar>
	int Style<Scalar>::GetNumSubstyles()
	{
		return (int)afStyle.dims(0);
	}

	template<typename Scalar>
	int Style<Scalar>::GetNumInducingSubstyles()
	{
		return (int)afInducingStyle.dims(0);
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetStyle()
	{
		return afStyle;
	}

	template<typename Scalar>
	std::map<std::string, int> Style<Scalar>::GetSubstyles()
	{
		return mSubStyles;
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetStyleExpanded()
	{
		return afStyle(afStyleIdx, af::span);
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetInducingStyleExpanded()
	{
		return afInducingStyle(afStyleInducingIdx, af::span);
	}

	template<typename Scalar>
	const af::array Style<Scalar>::GetStyleIndex()
	{
		return afStyleIdx;
	}

	template<typename Scalar>
	void Style<Scalar>::SetInducingStyleIndex(const af::array& indx)
	{
		afStyleInducingIdx = indx;
	}

	template<typename Scalar>
	void Style<Scalar>::GetFactorsExpanded(af::array& X1, af::array& X2)
	{
		X1 = afFactorX1(afStyleIdx, af::span);
		X2 = afFactorX2(afStyleIdx, af::span);
	}

	template<typename Scalar>
	std::string Style<Scalar>::GetName()
	{
		return sName;
	}

	template<typename Scalar>
	int Style<Scalar>::GetNumParameters()
	{
		return GetNumSubstyles() * GetNumSubstyles() * 2.0;
	}

	template<typename Scalar>
	int Style<Scalar>::GetNumInducingParameters()
	{
		return GetNumInducingSubstyles() * GetNumSubstyles();
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetParameters()
	{
		af::array param = flat(afFactorX1);
		return CommonUtil<Scalar>::Join(param, flat(af::log(afFactorX2) / 2.0));
	}

	template<typename Scalar>
	void Style<Scalar>::SetParameters(const af::array& param)
	{
		int iStart = 0, iEnd = GetNumSubstyles() * GetNumSubstyles();
		afFactorX1 = af::moddims(param(af::seq(iStart, iEnd - 1)), GetNumSubstyles(), GetNumSubstyles());

		iStart = iEnd; iEnd += GetNumSubstyles() * GetNumSubstyles();
		afFactorX2 = af::moddims(af::exp(2.0 * param(af::seq(iStart, iEnd - 1))), GetNumSubstyles(), GetNumSubstyles());
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetInducingParameters()
	{
		return flat(afInducingStyle);
	}

	template<typename Scalar>
	void Style<Scalar>::SetInducingParameters(const af::array& param)
	{
		afInducingStyle = moddims(param, GetNumInducingSubstyles(), GetNumSubstyles());
	}

	template<typename Scalar>
	void Style<Scalar>::UpdatePosterior(const af::array& postX1, const af::array& postX2)
	{
		af::array idx;
		afPosteriorX1 = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);
		afPosteriorX2 = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);
		for (auto i = 0; i < GetNumSubstyles(); i++)
		{
			idx = where(afStyleIdx == i)(0);
			afPosteriorX1(i, af::span) = postX1(idx, af::span);
			afPosteriorX2(i, af::span) = postX2(idx, af::span);
		}
		afStyle = afPosteriorX1 / afPosteriorX2;
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetGradientCollapsed(const af::array& factorGradient)
	{
		af::array collapsedGrad = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);

		for (auto i = 0; i < GetNumSubstyles(); i++)
			collapsedGrad(i, af::span) = af::sum(factorGradient(where(afStyleIdx == i), af::span));


		return flat(collapsedGrad);
	}

	template<typename Scalar>
	af::array Style<Scalar>::GetInducingGradientCollapsed(const af::array& factorGradient)
	{
		af::array collapsedGrad = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);

		for (auto i = 0; i < GetNumSubstyles(); i++)
			collapsedGrad(i, af::span) = af::sum(factorGradient(where(afStyleInducingIdx == i), af::span));


		return flat(collapsedGrad);
	}

	template<typename Scalar>
	void Style<Scalar>::InitFactors(const af::array& expandedX1, const af::array& expandedX2)
	{
		af::array idx;
		afFactorX1 = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);
		afFactorX2 = af::constant(0.0, GetNumSubstyles(), GetNumSubstyles(), m_dType);
		for (auto i = 0; i < GetNumSubstyles(); i++)
		{
			idx = where(afStyleIdx == i)(0);
			afFactorX1(i, af::span) = expandedX1(idx, af::span);
			afFactorX2(i, af::span) = expandedX2(idx, af::span);
		}
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	template<typename Scalar>
	GPLVMBaseModel<Scalar>::GPLVMBaseModel(const af::array& Y, int latentDimension, Scalar priorMean, Scalar priorVariance, LogLikType lType, XInit emethod)
		: GPBaseModel(Y, lType, ModelType::GPLVM), iq(latentDimension), dPriorMean(priorMean), dPriorVariance(priorVariance), dPriorX1(priorMean / priorVariance),
		dPriorX2(1.0 / priorVariance), eEmMethod(emethod), afPriorMean(), afPriorVariance(), afPriorMeanCav(), afPriorVarianceCav(), afGradMean(), afGradVariance(), afGradMeanCav(), afGradVarianceCav(),
		afPriorX1(), afPriorX2(), backConst(nullptr), mStyles(nullptr), bIsLatetsFixed(false)
	{
		if (!Y.isempty())
		{
			afFactorX1 = af::constant(0.0, iN, iq, m_dType);
			afFactorX2 = af::constant(0.0, iN, iq, m_dType);

			afPosteriorX1 = af::constant(0.0, iN, iq, m_dType);
			afPosteriorX2 = af::constant(0.0, iN, iq, m_dType);
		}
	}

	template<typename Scalar>
	GPLVMBaseModel<Scalar>::GPLVMBaseModel()
		: GPBaseModel(), iq(0), dPriorMean(0), dPriorVariance(1), eEmMethod(XInit::pca), bIsLatetsFixed(false)
	{
	}

	template<typename Scalar>
	GPLVMBaseModel<Scalar>::~GPLVMBaseModel()
	{
		if (backConst) delete backConst;
		if (mStyles) { mStyles->clear(); delete mStyles; }
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::Optimise(OptimizerType method, Scalar tol, bool reinit_hypers, int maxiter, int mb_size, LineSearchType lsType, bool disp, int* cycle)
	{
		/*if (mb_size > 0)
		{
			af::setSeed(time(NULL));
			afIndexes = af::round(af::randu(iN) * iN)(af::seq(mb_size));
		}*/

		/*if (backConst)
			maxiter /= 2;*/

		GPBaseModel::Optimise(method, tol, reinit_hypers, maxiter, mb_size, lsType, disp, cycle);

		/*FixLatents(true);

		GPBaseModel::Optimise(method, tol, reinit_hypers, maxiter, mb_size, lsType, disp, cycle);*/
	}

	template<typename Scalar>
	bool GPLVMBaseModel<Scalar>::Init(af::array& mx)
	{
		if (GetNumChildren() > 0)
		{
			for (uint i = 0; i < GetNumChildren(); i++)
			{
				af::array y;
				GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
				child.Init(y);
				afY = CommonUtil<Scalar>::Join(afY, y, 1);
			}

			iN = afY.dims(0);
			iD = afY.dims(1);

			if (iBatchSize >= iN || iBatchSize == 0)
			{
				iBatchSize = iN;
				afIndexes = af::seq(0, iN - 1);
			}
			else
			{
				af::setSeed(time(NULL));
				afIndexes = af::round(af::randu(iN) * iN)(af::seq(iBatchSize));
			}

			for (uint i = 0; i < GetNumChildren(); i++)
			{
				GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
				child.SetIndexes(afIndexes);
				child.SetBatchSize(iBatchSize);
			}
		}

		GPBaseModel::Init();

		int numNeighbours = 10;
		IEmbed* embed = nullptr;

		af::array vx(m_dType);

		switch (eEmMethod)
		{
		case XInit::pca:
			embed = new PCA();
			mx = embed->Compute(afY, iq);
			break;
		case XInit::isomap:
			embed = new Isomap(numNeighbours);
			mx = embed->Compute(afY, iq);
			break;
		case XInit::lle:
			embed = new LLE(numNeighbours);
			mx = embed->Compute(afY, iq);
			break;
		default:
			std::cout << "Embedding method not implemented." << std::endl;
			break;
		}
		if (embed != nullptr) delete embed;

		if (mStyles)
		{
			for (auto style = mStyles->begin(); style != mStyles->end(); style++)
				mx = CommonUtil<Scalar>::Join(mx, style->second.GetStyleExpanded(), 1);
		}

		vx = af::constant(0.1, mx.dims(), m_dType);

		if (backConst)
		{
			backConst->Init(afY, mx(af::span, af::seq(0, iq - 1)), afSegments);

			mx(af::span, af::seq(0, iq - 1)) = backConst->GetConstraintX();

			afFactorX1 = mx;
			afFactorX2 = vx;
		}
		else
		{
			// natural parameters computation
			afFactorX2 = 1.0 / vx;
			afFactorX1 = afFactorX2 * mx;
			afFactorX2 = /*af::exp(2.0 * (af::log(*/afFactorX2 - 1.0/*) / 2.0))*/;
		}

		if (mStyles)
		{
			int sStart = iq, sEnd = iq;
			for (auto style = mStyles->begin(); style != mStyles->end(); style++)
			{
				sStart = sEnd; sEnd += style->second.GetNumSubstyles();
				style->second.InitFactors(afFactorX1(af::span, af::seq(sStart, sEnd - 1)), afFactorX2(af::span, af::seq(sStart, sEnd - 1)));
			}
		}
			
		if (GetParent())
		{
			afPriorMean = af::constant(dPriorMean, iN, iq, m_dType);
			afPriorVariance = af::constant(dPriorVariance, iN, iq, m_dType);

			afPriorMeanCav = af::constant(dPriorMean, iN, iq, m_dType);
			afPriorVarianceCav = af::constant(dPriorVariance, iN, iq, m_dType);

			afPriorX1 = afPriorMean / afPriorVariance;
			afPriorX2 = 1.0 / afPriorVariance;

			afPriorX1Cav = afPriorMean / afPriorVariance;		//!< prior /f$x_1/f$
			afPriorX2Cav = 1.0 / afPriorVariance;
		}

		UpdateParametersInternal();

		return bInit;
	}

	template<typename Scalar>
	bool GPLVMBaseModel<Scalar>::Init()
	{
		bInit = GPBaseModel::Init();

		return bInit;
	}

	template<typename Scalar>
	int GPLVMBaseModel<Scalar>::GetNumParameters()
	{
		int numParam = 0;

		for (uint i = 0; i < GetNumChildren(); i++)
		{	
			GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
			numParam += child.GetNumParameters();
		}

		//if (GetNumChildren() == 0)
			numParam += GPBaseModel::GetNumParameters();

		if(!bIsLatetsFixed)
		{
			if (mStyles)
			{
				for (auto style = mStyles->begin(); style != mStyles->end(); style++)
					numParam += style->second.GetNumParameters();
			}

			if (backConst)
			{
				numParam += backConst->GetNumParameters();
				numParam += iBatchSize * iq;
			}
			else
				numParam += iBatchSize * iq * 2; // number of latents to be optimized
		}
			

		return numParam;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetParameters(const af::array& param)
	{
		int iStart = 0, iEnd = 0;
		for (uint i = 0; i < GetNumChildren(); i++)
		{
			GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
			iStart = iEnd;
			iEnd += child.GetNumParameters();
			child.SetParameters(param(af::seq(iStart, iEnd - 1)));
		}

		//if (GetNumChildren() == 0)
		{
			iStart = iEnd; iEnd += GPBaseModel::GetNumParameters();
			if (iStart != iEnd)
				GPBaseModel<Scalar>::SetParameters(param(af::seq(iStart, iEnd - 1)));
		}

		if(!bIsLatetsFixed)
		{
			iStart = iEnd;
			if (backConst)
			{
				iEnd += backConst->GetNumParameters();
				backConst->SetParameters(param(af::seq(iStart, iEnd - 1)));
				afFactorX1(afIndexes, af::seq(0, iq - 1)) = backConst->GetConstraintX();
			}
			else
			{
				iEnd += iBatchSize * iq;
				afFactorX1(afIndexes, af::seq(0, iq - 1)) = af::moddims(param(af::seq(iStart, iEnd - 1)), iBatchSize, iq);
			}
			iStart = iEnd; iEnd += iBatchSize * iq;
			afFactorX2(afIndexes, af::seq(0, iq - 1)) = af::moddims(af::exp(2.0 * param(af::seq(iStart, iEnd - 1))), iBatchSize, iq);

			if (mStyles)
			{
				int sStart = iq, sEnd = iq;
				af::array x1, x2;
				for (auto style = mStyles->begin(); style != mStyles->end(); style++)
				{
					sStart = sEnd; sEnd += style->second.GetNumSubstyles();
					iStart = iEnd; iEnd += style->second.GetNumParameters();
					style->second.SetParameters(param(af::seq(iStart, iEnd - 1)));

					style->second.GetFactorsExpanded(x1, x2);
					afFactorX1(afIndexes, af::seq(sStart, sEnd - 1)) = x1(afIndexes, af::span);
					afFactorX2(afIndexes, af::seq(sStart, sEnd - 1)) = x2(afIndexes, af::span);
				}
			}
			UpdateParametersInternal();
		}
	}

	template<typename Scalar>
	af::array GPLVMBaseModel<Scalar>::GetParameters()
	{
		af::array param = af::constant(0.0f, GPLVMBaseModel::GetNumParameters(), m_dType);

		int iStart = 0, iEnd = 0;
		for (uint i = 0; i < GetNumChildren(); i++)
		{
			GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
			iStart = iEnd;
			iEnd += child.GetNumParameters();
			param(af::seq(iStart, iEnd - 1)) = child.GetParameters();
		}

		//if (GetNumChildren() == 0)
		{
			iStart = iEnd; iEnd += GPBaseModel::GetNumParameters();
			if (iStart != iEnd)
				param(af::seq(iStart, iEnd - 1)) = GPBaseModel::GetParameters();
		}

		if(!bIsLatetsFixed)
		{
			iStart = iEnd;
			if (backConst)
			{
				iEnd += backConst->GetNumParameters();
				param(af::seq(iStart, iEnd - 1)) = backConst->GetParameters();
			}
			else
			{
				iEnd += iBatchSize * iq;
				param(af::seq(iStart, iEnd - 1)) = af::flat(afFactorX1(afIndexes, af::seq(0, iq - 1)));
			}
			iStart = iEnd; iEnd += iBatchSize * iq;
			param(af::seq(iStart, iEnd - 1)) = af::flat(af::log(afFactorX2(afIndexes, af::seq(0, iq - 1))) / 2.0);

			if (mStyles)
			{
				for (auto style = mStyles->begin(); style != mStyles->end(); style++)
				{
					iStart = iEnd; iEnd += style->second.GetNumParameters();
					param(af::seq(iStart, iEnd - 1)) = style->second.GetParameters();
				}
			}
		}

		return param;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::UpdateParameters()
	{
		GPBaseModel<Scalar>::UpdateParameters();
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::FixKernelParameters(bool isfixed)
	{
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::FixInducing(bool isfixed)
	{
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::FixLatents(bool isFixed)
	{
		bIsLatetsFixed = isFixed;
	}

	template<typename Scalar>
	af::array GPLVMBaseModel<Scalar>::GetMeanGradient()
	{
		return afGradMean(afIndexes, af::span);
	}

	template<typename Scalar>
	af::array GPLVMBaseModel<Scalar>::GetVarGradient()
	{
		return afGradVariance(afIndexes, af::span);
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetPrior(const af::array mean, const af::array var)
	{
		afPriorMean(afIndexes, af::span) = mean;
		afPriorVariance(afIndexes, af::span) = var;

		afPriorX1(afIndexes, af::span) = mean / var;
		afPriorX2(afIndexes, af::span) = 1.0 / var;

		UpdateParametersInternal();
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetPriorCavity(const af::array meanCav, const af::array varCav)
	{
		afPriorMeanCav(afIndexes, af::span) = meanCav;
		afPriorVarianceCav(afIndexes, af::span) = varCav;

		afPriorX1Cav(afIndexes, af::span) = meanCav / varCav;
		afPriorX2Cav(afIndexes, af::span) = 1.0 / varCav;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetLatentGradient(const af::array& dmParent, const af::array& dvParent)
	{
		afGradMean = dmParent;
		afGradVariance = dvParent;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetLatentGradientCavity(const af::array& dmParent, const af::array& dvParent)
	{
		afGradMeanCav = dmParent;
		afGradVarianceCav = dvParent;
	}

	template<typename Scalar>
	int GPLVMBaseModel<Scalar>::GetLatentDimension()
	{
		return iq;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetBackConstraint(IBackconstraint<Scalar>* constraint)
	{
		backConst = constraint;
	}

	template<typename Scalar>
	IBackconstraint<Scalar>* GPLVMBaseModel<Scalar>::GetBackConstraint()
	{
		return backConst;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::SetStyles(std::map<std::string, Style<Scalar>>* styles)
	{
		mStyles = styles;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::AddStyle(Style<Scalar> style)
	{
		if (!mStyles) mStyles = new std::map<std::string, Style<Scalar>>();
		mStyles->insert(std::make_pair(style.GetName(), style));
	}

	template<typename Scalar>
	std::map<std::string, Style<Scalar>>* GPLVMBaseModel<Scalar>::GetStyles()
	{
		return mStyles;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::PosteriorLatents(af::array& mx, af::array& vx)
	{
		vx = 1.0 / afPosteriorX2(afIndexes, af::span);
		mx = afPosteriorX1(afIndexes, af::span) / afPosteriorX2(afIndexes, af::span);
	}

	template<typename Scalar>
	af::array GPLVMBaseModel<Scalar>::PosteriorGradientLatents(const af::array& dmx, const af::array& dvx)
	{
		af::array gradX1, gradX2, gradX1Back, gradX2Back, gradStyleFlat;
		af::array gradX;

		int iStart = 0, iEnd, numParam = 0;

		if (backConst)
		{
			iEnd = backConst->GetNumParameters();
			gradX1Back = backConst->BackconstraintGradient(dmx(af::span, af::seq(0, iq - 1)));
			gradX2Back = backConst->BackconstraintGradient(dvx(af::span, af::seq(0, iq - 1)) * 2.0 * afFactorX2(afIndexes, af::seq(0, iq - 1)));
			//gradX2Back *= 2.0 * afFactorX2(afIndexes, af::seq(0, iq - 1));

			numParam += /*iBatchSize * iq + */backConst->GetNumParameters();
			//gradX = constant(0.0, iBatchSize * iq + backConst->GetNumParameters(), m_dType);
		}
		else
		{
			iEnd = iBatchSize * iq;
			gradX1 = dmx / afPosteriorX2(afIndexes, af::span);
			gradX2 = -dmx * afPosteriorX1(afIndexes, af::span) / pow(afPosteriorX2(afIndexes, af::span), 2.0) - dvx / pow(afPosteriorX2(afIndexes, af::span), 2.0);
			gradX2 *= 2.0 * afFactorX2(afIndexes, af::span);

			numParam += iBatchSize * iq * 2;
		}
		
		if (mStyles)
		{
			if (backConst)
			{
				gradX1 = dmx;
				gradX2 = dvx;
				gradX2 *= 2.0 * afFactorX2(afIndexes, af::span);
			}

			int sStart = iq, sEnd = iq;
			for (auto style = mStyles->begin(); style != mStyles->end(); style++)
			{
				sStart = sEnd; sEnd += style->second.GetNumSubstyles();

				gradStyleFlat = CommonUtil<Scalar>::Join(gradStyleFlat, style->second.GetGradientCollapsed(gradX1(af::span, af::seq(sStart, sEnd - 1))));
				gradStyleFlat = CommonUtil<Scalar>::Join(gradStyleFlat, style->second.GetGradientCollapsed(gradX2(af::span, af::seq(sStart, sEnd - 1))));
				numParam += style->second.GetNumParameters();
			}

			gradX1 = gradX1(af::span, af::seq(0, iq - 1));
			gradX2 = gradX2(af::span, af::seq(0, iq - 1));
		}

		if (backConst)
		{
			gradX1 = gradX1Back;
			gradX2 = gradX2Back;
		}

		gradX = CommonUtil<Scalar>::Join(gradX, flat(gradX1));
		gradX = CommonUtil<Scalar>::Join(gradX, flat(gradX2));

		if (mStyles)
		{
			gradX = CommonUtil<Scalar>::Join(gradX, gradStyleFlat);
		}
		

		return gradX;
	}

	template<typename Scalar>
	af::array GPLVMBaseModel<Scalar>::LatentGradient(const af::array& dm, const af::array& dv)
	{
		af::array gradX1, gradX2, gradX;

		gradX1 = dm / dv;
		gradX2 = 1.0 / dv;

		gradX = CommonUtil<Scalar>::Join(gradX, flat(gradX1));
		gradX = CommonUtil<Scalar>::Join(gradX, flat(gradX2));

		return gradX;
	}

	template<typename Scalar>
	void GPLVMBaseModel<Scalar>::UpdateParametersInternal()
	{
		GPBaseModel<Scalar>::UpdateParameters();

		if (GetNumChildren() > 0)
		{
			afY = af::array();
			af::array mx, vx;
			for (uint i = 0; i < GetNumChildren(); i++)
			{
				GPLVMBaseModel<Scalar>& child = dynamic_cast<GPLVMBaseModel<Scalar>&>(*GetChild(i));
				child.PosteriorLatents(mx, vx);
				afY = CommonUtil<Scalar>::Join(afY, mx, 1);
			}
		}

		if (backConst)
		{
			afPosteriorX1 = afFactorX1 / afFactorX2;
			afPosteriorX2 = 1.0 / afFactorX2;
		}
		else
		{
			if (GetParent())
			{
				afPosteriorX1 = afPriorX1 + afFactorX1;
				afPosteriorX2 = afPriorX2 + afFactorX2;
			}
			else
			{
				afPosteriorX1 = dPriorX1 + afFactorX1;
				afPosteriorX2 = dPriorX2 + afFactorX2;
			}
		}

		if (mStyles)
		{
			int sStart = iq, sEnd = iq;
			for (auto style = mStyles->begin(); style != mStyles->end(); style++)
			{
				sStart = sEnd; sEnd += style->second.GetNumSubstyles();

				style->second.UpdatePosterior(afPosteriorX1(af::span, af::seq(sStart, sEnd - 1)), afPosteriorX2(af::span, af::seq(sStart, sEnd - 1)));
			}
		}

		
	}
}