/**
File:		MachineLearning/FactorGraph/Node/FgGPNode.cpp

Author:		
Email:		
Site:       

Copyright (c) 2020 . All rights reserved.
*/

#include <NeMachineLearningPCH.h>
#include <MachineLearning/FgGPNode.h>

namespace NeuralEngine::MachineLearning
{
	template class GPNode<float>;
	template class GPNode<double>;

	template<typename Scalar>
	GPNode<Scalar>::GPNode()
		: mParent(nullptr), mChild()
	{
	}

	template<typename Scalar>
	GPNode<Scalar>::~GPNode()
	{
		for (auto& child : mChild)
		{
			if (child)
			{
				child->SetParent(nullptr);
				child = nullptr;
			}
		}
	}

	template<typename Scalar>
	int GPNode<Scalar>::GetNumChildren() const
	{
		return static_cast<int>(mChild.size());
	}

	template<typename Scalar>
	int GPNode<Scalar>::AttachChild(std::shared_ptr<GPNode> const& child)
	{
		if (!child)
		{
			LogError("You cannot attach null children to a node.");
			return -1;
		}

		if (child->GetParent())
		{
			LogError("The child already has a parent.");
			return -1;
		}

		child->SetParent(this);

		// Insert the child in the first available slot (if any).
		int i = 0;
		for (auto& current : mChild)
		{
			if (!current)
			{
				current = child;
				return i;
			}
			++i;
		}

		// All slots are used, so append the child to the array.
		int const numChildren = static_cast<int>(mChild.size());
		mChild.push_back(child);
		return numChildren;
	}

	template<typename Scalar>
	int GPNode<Scalar>::DetachChild(std::shared_ptr<GPNode> const& child)
	{
		if (child)
		{
			int i = 0;
			for (auto& current : mChild)
			{
				if (current == child)
				{
					current->SetParent(nullptr);
					current = nullptr;
					return i;
				}
				++i;
			}
		}
		return -1;
	}

	template<typename Scalar>
	std::shared_ptr<GPNode<Scalar>> GPNode<Scalar>::DetachChildAt(int i)
	{
		if (0 <= i && i < static_cast<int>(mChild.size()))
		{
			std::shared_ptr<GPNode<Scalar>> child = mChild[i];
			if (child)
			{
				child->SetParent(nullptr);
				mChild[i] = nullptr;
			}
			return child;
		}
		return nullptr;
	}

	template<typename Scalar>
	void GPNode<Scalar>::DetachAllChildren()
	{
		for (auto& current : mChild)
		{
			DetachChild(current);
		}
	}

	template<typename Scalar>
	std::shared_ptr<GPNode<Scalar>> GPNode<Scalar>::SetChild(int i, std::shared_ptr<GPNode> const& child)
	{
		if (child)
		{
			LogAssert(!child->GetParent(), "The child already has a parent.");
		}

		int const numChildren = static_cast<int>(mChild.size());
		if (0 <= i && i < numChildren)
		{
			// Remove the child currently in the slot.
			std::shared_ptr<GPNode<Scalar>> previousChild = mChild[i];
			if (previousChild)
			{
				previousChild->SetParent(nullptr);
			}

			// Insert the new child in the slot.
			if (child)
			{
				child->SetParent(this);
			}

			mChild[i] = child;
			return previousChild;
		}

		// The index is out of range, so append the child to the array.
		if (child)
		{
			child->SetParent(this);
		}
		mChild.push_back(child);
		return nullptr;
	}

	template<typename Scalar>
	std::shared_ptr<GPNode<Scalar>> GPNode<Scalar>::GetChild(int i)
	{
		if (0 <= i && i < static_cast<int>(mChild.size()))
		{
			return mChild[i];
		}
		return nullptr;
	}

	template<typename Scalar>
	GPNode<Scalar>* GPNode<Scalar>::GetParent()
	{
		return mParent;
	}

	template<typename Scalar>
	void GPNode<Scalar>::SetParent(GPNode<Scalar>* parent)
	{
		mParent = parent;
	}
}