/**
File:		MachineLearning/Embed/FgLLE.cpp

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#include <NeMachineLearningPCH.h>
#include <MachineLearning/FgIsomap.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		Isomap::Isomap(int numNeighbours)
			: _numNeighbours(numNeighbours)
		{

		}

		Isomap::~Isomap() { }

		af::array Isomap::Compute(af::array& inM, int q)
		{
			af::array X = af::constant(0.0, inM.dims(0), q);

			af::array D = L2_distance(inM.T(), inM.T(), true);

			std::cout << "Isomap running on " << inM.dims(0) << " points in " << inM.dims(1) << " dimensions." << std::endl;

			Xstruct xstruct = isomap(D, _numNeighbours, q);

			if (xstruct.index.elements() != inM.dims(0))
			{
				// We don't really deal with this problem correctly here ...
				std::cout << "Isomap graph is not fully connected!" << std::endl;
			}

			X(xstruct.index, af::span) = xstruct.coords[q - 1].T();

			return X;
		}

		Isomap::Xstruct Isomap::isomap(af::array& D, int n_size, int q)
		{
			Xstruct X;

			int N = D.dims(0);
			int K = n_size;

			double INF = (double)(1000 * af::max<double>(af::max(D)) * N);  // effectively infinite distance


			af::array dims = af::constant(0.0, q, 1);
			gfor(af::seq i, q)
				dims(i) = i;

			int comp = 0;
			bool verbose = true;

			//X.coords = ILMath.cell(dims.dims(0), 1);
			af::array R = af::constant(0.0, dims.dims(0));

			// Step 1: Construct neighborhood graph
			std::cout << "\t-->Constructing neighborhood graph." << std::endl;

			af::array perm, sorted, tmp;
			af::sort(sorted, perm, D);
			for (int i = 0; i < N; i++)
			{
				tmp = perm(af::seq(1 + K, af::end), i);
				D(i, tmp) = INF;
			}


			D = af::min(D, D.T());    // Make sure distance matrix is symmetric

										//if (overlay)
										//    int8 E = int8(1-(D==INF));  //  Edge information for subsequent graph overlay

										// Finite entries in D now correspond to distances between neighboring points. 
										// Infinite entries (really, equal to INF) in D now correspond to 
										//   non-neighoring points. 

										// Step 2: Compute shortest paths
			std::cout << "\t-->Computing shortest paths..." << std::endl;

			// We use Floyd's algorithm, which produces the best performance in Matlab. 
			// Dijkstra's algorithm is significantly more efficient for sparse graphs, 
			// but requires for-loops that are very slow to run in Matlab.  A significantly 
			// faster implementation of Isomap that calls a MEX file for Dijkstra's 
			// algorithm can be found in isomap2.m (and the accompanying files
			// dijkstra.c and dijkstra.dll). 

			for (int k = 0; k < N; k++)
			{
				D = af::min(D, af::tile(D(af::span, k), 1, N) + af::tile(D(af::span, k), 1, N).T());
				if (verbose && (k % 20 == 0) && k != 0)
					std::cout << "\t\tIteration: " << k << std::endl;
			}

			// Remove outliers from graph
			std::cout << "\n\t-->Checking for outliers." << std::endl;
			af::array n_connect = af::sum(!(D == INF)); // number of points each point connects to
			af_print(n_connect);
			af::array firsts;
			af::min(tmp, firsts, D == INF);				// first point each point connects to
			//af_print(D == INF);
			af::array comps = af::setUnique(firsts);    // represent each connected component once
			af::array size_comps = n_connect(comps);    // size of each connected component

			af::sort(sorted, perm, size_comps);         // sort connected components by size

			perm = perm(af::seq(af::end, -1, 0));

			comps = comps(perm);

			size_comps = size_comps(perm);
			int n_comps = comps.dims(1);               // number of connected components
			if (comp > n_comps)
				comp = 0;                              // default: use largest component

			std::cout << "\nNumber of connected components in graph: " << n_comps << std::endl;
			std::cout << "Embedding component " << comp << " with " << size_comps(comp).scalar<float>() << " points." << std::endl;

			X.index = (firsts == comps(comp));

			D = D(X.index, X.index);
			N = X.index.dims(1);

			// Step 3: Construct low-dimensional embeddings (Classical MDS)
			std::cout << "\n\t-->Constructing low-dimensional embeddings (Classical MDS)." << std::endl;

			//opt.disp = 0;

			/*arma::mat src = AfArma::ArrayToMat(-.5 * (af::pow(D, 2)
				- af::matmulTN(af::sum(af::pow(D, 2)), af::constant(1, 1, N) / N)
				- af::matmul(af::constant(1, N, 1), af::sum(af::pow(D, 2)) / N)
				+ af::sum(af::sum(af::pow(D, 2))) / (N + N)));
			arma::vec eigval;
			arma::mat eigvec;

			arma::eig_sym(eigval, eigvec, src);

			af::array val = AfArma::MatToArray(eigval);
			af::array vec = AfArma::MatToArray(eigvec);

			af::array index;
			af::sort(sorted, index, -val.T());

			val = -sorted;
			vec = vec(af::span, index);

			D = af::tile(D, N * N, 1);
			af::array r2;
			int ind;
			for (int di = 0; di < dims.dims(0); di++)
			{
				ind = dims(di).scalar<float>();
				if (ind <= N)
				{
					tmp = seq(0, ind);
					X.coords[di] = vec(af::span, tmp) *
						af::matmulNT(af::constant(1, N, 1), af::sqrt(val(tmp))).T();

					r2 = 1 - af::pow(CommonUtil<double>::CorrelationCoefficients(af::tile(L2_distance(X.coords[di],
						X.coords[di], false), N * N, 1), D), 2);
					R(di) = r2(1, 0);
					if (verbose)
						std::cout << "  Isomap on " << N << " points with dimensionality " << ind << " --> residual variance = " << R(di).scalar<float>() << std::endl;

				}
			}*/
			return X;
		}

		af::array Isomap::L2_distance(af::array& a, af::array& b, bool df)
		{
			af::array aa = af::sum(a * a);
			af::array bb = af::sum(b * b);
			af::array ab = af::matmulTN(a, b);

			af::array d = af::sqrt(af::tile(aa.T(), 1, bb.dims(1)) +
				af::tile(bb, aa.dims(1), 1) - 2 * ab);

			// force 0 on the diagonal? 
			if (df)
				d = d *(1 - af::identity(d.dims(0), d.dims(1)));

			return d;
		}
	}
}