/**
File:		MachineLearning/Kernel/FgRBFAccelerationKernel<Scalar>.cpp

Author:		Nick Taubert
Email:		nick.taubert@uni-tuebingen.de
Site:       http://www.compsens.uni-tuebingen.de/

Copyright (c) 2017 CompSens. All rights reserved.
*/

#include <NeEnginePCH.h>
#include <Core/NeLogger.h>
#include <MachineLearning/FgRBFAccelerationKernel.h>
#include <MachineLearning/CommonUtil.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		template class RBFAccelerationKernel<float>;
		template class RBFAccelerationKernel<double>;

		template<typename Scalar>
		RBFAccelerationKernel<Scalar>::RBFAccelerationKernel()
			: IKernel<Scalar>(eRBFAccelerationKernel, 3), dVariance(1.0), dInvScale1(1.0), dInvScale2(1.0)
		{
		}

		template<typename Scalar>
		RBFAccelerationKernel<Scalar>::~RBFAccelerationKernel() { }

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::ComputeKernelMatrix(const af::array& inX1, const af::array& inX2, af::array& outMatrix)
		{
			int q = inX1.dims(1) / 2;

			outMatrix = dVariance * af::exp(-CommonUtil<Scalar>::SquareDistance(inX1(af::span, af::seq(0, q - 1)), inX2(af::span, af::seq(0, q - 1)))
				* 0.5f * dInvScale1 - CommonUtil<Scalar>::SquareDistance(inX1(af::span, af::seq(q, 2 * q - 1)), inX2(af::span, af::seq(q, 2 * q - 1))) * 0.5f * dInvScale2);
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::ComputeDiagonal(const af::array& inX, af::array& outDiagonal)
		{
			outDiagonal = af::tile(dVariance, inX.dims(0), 1);
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::LogLikGradientX(const af::array& inX, const af::array& indL_dK, af::array& outdL_dX)
		{
			outdL_dX = af::constant(0.0f, inX.dims(), (m_dType));
			af::array dK_dX(m_dType);
			for (int q = 0; q < outdL_dX.dims(1); q++)
			{
				GradX(inX, inX, q, dK_dX);
				outdL_dX(af::span, q) = 2.0f * af::sum(indL_dK * dK_dX, 1) - af::diag(indL_dK) * af::diag(dK_dX);
			}
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::LogLikGradientX(const af::array& inXu, const af::array& indL_dKuu, const af::array& inX, const af::array& indL_dKuf, af::array& outdL_dXu, af::array& outdL_dX)
		{
			outdL_dXu = af::constant(0.0f, inXu.dims(), (m_dType));
			outdL_dX = af::constant(0.0f, inX.dims(), (m_dType));

			af::array dK_dX_u(m_dType); // overrider for dX and dXu
			for (int q = 0; q < outdL_dXu.dims(1); q++)
			{
				// dL_dKuu_dXu
				GradX(inXu, inXu, q, dK_dX_u);
				outdL_dXu(af::span, q) = 2 * af::sum(indL_dKuu * dK_dX_u, 1) - af::diag(indL_dKuu) * af::diag(dK_dX_u);

				// dL_dKuf_dXu
				GradX(inXu, inX, q, dK_dX_u);
				outdL_dXu(af::span, q) += af::sum(indL_dKuf * dK_dX_u, 1);

				// dL_dKuf_dX
				GradX(inX, inXu, q, dK_dX_u);
				outdL_dX(af::span, q) = af::sum(indL_dKuf.T() * dK_dX_u, 1);
			}
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::LogLikGradientParam(const af::array& inX1, const af::array& inX2, const af::array& indL_dK, af::array& outdL_dParam)
		{
			af::array K(m_dType), n2(m_dType), tmp(m_dType);
			ComputeKernelMatrix(inX1, inX2, K);

			int q = inX1.dims(1) / 2;

			n2 = CommonUtil<Scalar>::SquareDistance(inX1(af::span, af::seq(0, q - 1)), inX2(af::span, af::seq(0, q - 1)));
			tmp = -0.5 * K * n2;
			outdL_dParam = af::array(1, 3, (m_dType));
			outdL_dParam(1) = af::sum(af::sum(indL_dK * tmp)) * dInvScale1; // consider log space

			n2 = CommonUtil<Scalar>::SquareDistance(inX1(af::span, af::seq(q, 2 * q - 1)), inX2(af::span, af::seq(q, 2 * q - 1)));
			tmp = -0.5 * K * n2;
			outdL_dParam(2) = af::sum(af::sum(indL_dK * tmp)) * dInvScale2; // consider log space

			tmp = K / dVariance;
			outdL_dParam(0) = af::sum(af::sum(indL_dK * tmp)) * dVariance; // consider log space
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::GradX(const af::array& inX1, const af::array& inX2, int q, af::array& outdK_dX)
		{
			int numData = inX1.dims(0);
			int numData2 = inX2.dims(0);

			af::array K1 = af::tile(inX1(af::span, q), 1, numData2);
			af::array K2 = af::tile(inX2(af::span, q).T(), numData, 1);

			af::array K(m_dType);
			ComputeKernelMatrix(inX1, inX2, K);

			if (q < inX1.dims(1) / 2)
				if (K1.dims(0) == K.dims(0))
					outdK_dX = -dInvScale1 * (K1 - K2) * K;
				else
					outdK_dX = -dInvScale1 * (K1 - K2) * K.T();
			else
				if (K1.dims(0) == K.dims(0))
					outdK_dX = -dInvScale2 * (K1 - K2) * K;
				else
					outdK_dX = -dInvScale2 * (K1 - K2) * K.T();
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::DiagGradX(const af::array& inX, af::array& outDiagdK_dX)
		{
			outDiagdK_dX = af::constant(0.0, inX.dims(), (m_dType));
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::DiagGradParam(const af::array& inX, const af::array& inCovDiag, af::array& outDiagdK_dParam)
		{
			outDiagdK_dParam = af::array(1, 3, (m_dType));

			outDiagdK_dParam(0) = 0.0;
			outDiagdK_dParam(1) = 0.0;
			outDiagdK_dParam(2) = af::sum(inCovDiag);
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::SetParameters(const af::array & param)
		{
			dVariance = param(0).scalar<Scalar>();
			dInvScale1 = param(1).scalar<Scalar>();
			dInvScale2 = param(2).scalar<Scalar>();
		}

		template<typename Scalar>
		af::array RBFAccelerationKernel<Scalar>::GetParameters()
		{
			af::array param = constant(0.0f, 1, GetNumParameter(), (m_dType));
			param(0) = dVariance;
			param(1) = dInvScale1;
			param(1) = dInvScale2;
			return param;
		}

		template<typename Scalar>
		void RBFAccelerationKernel<Scalar>::Psi1Derivative(const af::array & inPsi1, const af::array & indL_dpsi1, const af::array & inZ, const af::array & inMu, const af::array & inSu, af::array & outdL_dParam, af::array & outdL_dXu, af::array * outdL_dX)
		{
		}
	}
}
