/**
File:		MachineLearning/Optimization/NonlinearObjectiveFunction.cpp

Author:		
Email:		
Site:       

Copyright (c) 2017 . All rights reserved.
*/

#include <NeMachineLearningPCH.h>
#include <MachineLearning/NonlinearObjectiveFunction.h>

namespace NeuralEngine
{
	namespace MachineLearning
	{
		template<typename Scalar>
		NonlinearObjectiveFunction<Scalar>::NonlinearObjectiveFunction()
			: _numVariables(0), _function(), afUpperBound(), afLowerBound(), _dtype(CommonUtil<Scalar>::CheckDType())//, _gradient(_dtype)
		{
			_gradient = af::array(_dtype);
		}

		template<typename Scalar>
		NonlinearObjectiveFunction<Scalar>::NonlinearObjectiveFunction(int numberOfVariables)
			: _numVariables(0), _function(), afUpperBound(), afLowerBound(), _dtype(CommonUtil<Scalar>::CheckDType())//, _gradient(_dtype)
		{
			_gradient = af::array(_dtype);

			_numVariables = numberOfVariables;

			af::array _bound = af::constant(af::Inf, numberOfVariables, _dtype);
			afUpperBound = _bound;
			afLowerBound = -_bound;

			/*for (int i = 0; i < numberOfVariables; i++)
			{
				std::string name = "x" + i;
				_variables->insert(std::pair<std::string, int>(name, i));
				_indices->insert(std::pair<int, std::string>(i, name));
			}*/
		}

		template<typename Scalar>
		NonlinearObjectiveFunction<Scalar>::NonlinearObjectiveFunction(int numberOfVariables, std::function<Scalar(const af::array&, af::array&)> function)
			: NonlinearObjectiveFunction(numberOfVariables)
		{
			_function = function;
		}

		template<typename Scalar>
		NonlinearObjectiveFunction<Scalar>::~NonlinearObjectiveFunction()
		{
		}

		template<typename Scalar>
		std::function<Scalar(const af::array&, af::array&)> NonlinearObjectiveFunction<Scalar>::GetFunction()
		{
			return _function;
		}

		template<typename Scalar>
		void NonlinearObjectiveFunction<Scalar>::SetFunction(std::function<Scalar(const af::array&, af::array&)> func)
		{
			_function = func;
		}

		template<typename Scalar>
		Scalar NonlinearObjectiveFunction<Scalar>::Value(const af::array & inX)
		{
			return _function(inX, _gradient);
		}

		template<typename Scalar>
		af::array NonlinearObjectiveFunction<Scalar>::Gradient(const af::array & x)
		{
			return _gradient;
		}

		template<typename Scalar>
		int NonlinearObjectiveFunction<Scalar>::GetNumberOfVariables()
		{
			return _numVariables;
		}

		template<typename Scalar>
		void NonlinearObjectiveFunction<Scalar>::CheckGradient(const af::array& probe)
		{
			af::array original = probe.copy();
			af::array actual_grad;
			_function(probe, actual_grad);
			af::array expected_grad = FiniteGradient(probe, 3);

			if (CommonUtil<Scalar>::IsEqual(actual_grad, probe)) LogError("The gradient function should not return the parameter vector.");
			if (actual_grad.dims(0) != probe.dims(0)) LogError("The gradient vector should have the same length as the number of parameters.");

			for (uint d = 0; d < probe.dims(0); ++d)
			{
				Scalar actGradD = actual_grad(d).scalar<Scalar>();
				Scalar expectGradD = expected_grad(d).scalar<Scalar>();
				Scalar scale = std::max(static_cast<Scalar>(std::max(fabs(actGradD), fabs(expectGradD))), Scalar(1.));
				if (fabs(actGradD - expectGradD) > 1e-2 * scale)
					std::cout << "Actual Gradient " << d << " doesn't match expected one.";
			}
		}

		template<typename Scalar>
		af::array NonlinearObjectiveFunction<Scalar>::FiniteGradient(const af::array & x, int accuracy)
		{
			// accuracy can be 0, 1, 2, 3
			const Scalar eps = 2.2204e-6;
			static const std::array<std::vector<Scalar>, 4> coeff =
			{ { { 1, -1 },{ 1, -8, 8, -1 },{ -1, 9, -45, 45, -9, 1 },{ 3, -32, 168, -672, 672, -168, 32, -3 } } };
			static const std::array<std::vector<Scalar>, 4> coeff2 =
			{ { { 1, -1 },{ -2, -1, 1, 2 },{ -3, -2, -1, 1, 2, 3 },{ -4, -3, -2, -1, 1, 2, 3, 4 } } };
			static const std::array<Scalar, 4> dd = { 2, 12, 60, 840 };

			af::array grad = af::constant(0.0, x.dims(0), _dtype);
			af::array xx = x.copy();

			const int innerSteps = 2 * (accuracy + 1);
			const Scalar ddVal = dd[accuracy] * eps;

			for (uint d = 0; d < x.dims(0); d++) 
			{
				for (int s = 0; s < innerSteps; ++s)
				{
					af::array tmp = xx(d);
					xx(d) += coeff2[accuracy][s] * eps;
					grad(d) += coeff[accuracy][s] * Value(xx);
					xx(d) = tmp;
				}
				grad(d) /= ddVal;
			}

			return grad;
		}

		template<typename Scalar>
		af::array & NonlinearObjectiveFunction<Scalar>::LowerBound()
		{
			return afLowerBound;
		}

		template<typename Scalar>
		af::array & NonlinearObjectiveFunction<Scalar>::UpperBound()
		{
			return afUpperBound;
		}

		template class NonlinearObjectiveFunction<float>;
		template class NonlinearObjectiveFunction<double>;
	}
}